--[[
    Zero Operating Costs (FS25)
    Forces all production / building / equipment upkeep and operating
    costs effectively to 0.

    Behavior:
    - Zeroes cost fields on productions / placeables / vehicles
      (so other mods' values get overwritten back to 0).
    - Blocks money charges for PROPERTY_MAINTENANCE,
      VEHICLE_RUNNING_COSTS, and PRODUCTION_COSTS at Farm.changeBalance.
    - Suppresses HUD money popups for those charges by patching
      g_currentMission.hud:addMoneyChange().

    Notes:
    - This mod focuses on *running/maintenance* costs, not purchases.
    - If GIANTS renames / changes MoneyType enums in a future patch,
      you may only need to adjust isBlockedMoneyType().
--]]

print("ZeroOperatingCosts.lua loaded (pre-init)")

ZeroOperatingCosts = {}
local ZOC = ZeroOperatingCosts

ZOC.modName = g_currentModName or "FS25_ZeroOperatingCosts"

----------------------------------------------------------------------
-- Configuration
----------------------------------------------------------------------
ZOC.debug = false  -- set to false once you're happy

ZOC.CONFIG = {
    -- How often to re-sweep and re-zero all known costs (ms real time).
    -- This is mainly to fight other mods that re-apply upkeep values.
    scanIntervalMs = 5000,         -- 5 seconds

    -- Toggle which categories get swept:
    sweepProductions = true,
    sweepPlaceables = true,
    sweepVehicles   = true,
}

----------------------------------------------------------------------
-- Logging helper
----------------------------------------------------------------------
local function zocLog(...)
    if not ZOC.debug then return end
    local parts = {}
    for i, v in ipairs({...}) do
        parts[i] = tostring(v)
    end
    print("[" .. (ZOC.modName or "ZeroOperatingCosts") .. "] " .. table.concat(parts, " "))
end

----------------------------------------------------------------------
-- Helper: decide which money types we want to block
----------------------------------------------------------------------
local function isBlockedMoneyType(moneyType)
    if MoneyType == nil or moneyType == nil then
        return false
    end

    -- FS-style enums; if FS25 renames them, adjust this function.
    return moneyType == MoneyType.PROPERTY_MAINTENANCE
        or moneyType == MoneyType.VEHICLE_RUNNING_COSTS
        or moneyType == MoneyType.PRODUCTION_COSTS
end

----------------------------------------------------------------------
-- Shared field lists
----------------------------------------------------------------------

-- Any generic “upkeep” / “maintenance” fields on objects / placeables /
-- productionPoints
ZOC.OBJECT_UPKEEP_FIELDS = {
    "dailyUpkeep",
    "maintenanceCost",
    "maintenancePerDay",
    "upkeepPerDay",
    "upkeep",
    "runningCosts",
    "runningCostPerHour",
    "runningCostPerMinute",
    "costsPerDay",
    "costsPerHour",
    "costsPerMinute",
    "costsPerMonth",
    "costsPerCycle"
}

-- Things usually found on individual production entries
ZOC.PRODUCTION_FIELDS = {
    "costsPerActiveMonth",
    "costsPerActiveHour",
    "costsPerActiveMinute",
    "costsPerMinute",
    "costsPerHour",
    "costsPerDay",
    "costsPerCycle",
    "maintenanceCost",
    "runningCosts"
}

-- Vehicle-centric cost fields
ZOC.VEHICLE_FIELDS = {
    "dailyUpkeep",
    "runningCosts",
    "operatingCost",
    "operatingCostPerHour",
    "operatingCostPerSecond",
    "costsPerHour",
    "costsPerDay"
}

----------------------------------------------------------------------
-- Zero building-level / object-level upkeep fields
----------------------------------------------------------------------
function ZOC.zeroUpkeepFields(obj)
    if obj == nil then return end

    local touched = false

    for _, f in ipairs(ZOC.OBJECT_UPKEEP_FIELDS) do
        local v = obj[f]
        if v ~= nil and type(v) == "number" and v ~= 0 then
            zocLog("Zeroing object field", f, "from", v, "to 0")
            obj[f] = 0
            touched = true
        end
    end

    -- Common maintenance table pattern (if present)
    local m = obj.maintenance
    if m ~= nil and type(m) == "table" then
        local v = m.costPerDay
        if type(v) == "number" and v ~= 0 then
            zocLog("Zeroing object.maintenance.costPerDay from", v, "to 0")
            m.costPerDay = 0
            touched = true
        end
    end

    if touched then
        zocLog("Finished zeroing upkeep for object", tostring(obj))
    end
end

----------------------------------------------------------------------
-- Zero per-production-line running costs
----------------------------------------------------------------------
function ZOC.zeroProductionCosts(production)
    if production == nil then return end

    local idStr = tostring(production.id or "?")
    local touched = false

    for _, f in ipairs(ZOC.PRODUCTION_FIELDS) do
        local v = production[f]
        if v ~= nil and type(v) == "number" and v ~= 0 then
            zocLog("Zeroing production line", idStr, "field", f, "from", v, "to 0")
            production[f] = 0
            touched = true
        end
    end

    if touched then
        zocLog("Finished zeroing production line", idStr)
    end
end

----------------------------------------------------------------------
-- Zero vehicle / equipment running-cost-related fields
----------------------------------------------------------------------
function ZOC.zeroVehicleCosts(vehicle)
    if vehicle == nil then return end

    local touched = false

    -- Generic vehicle fields
    for _, f in ipairs(ZOC.VEHICLE_FIELDS) do
        local v = vehicle[f]
        if v ~= nil and type(v) == "number" and v ~= 0 then
            zocLog("Zeroing vehicle field", f, "from", v, "to 0 on", tostring(vehicle))
            vehicle[f] = 0
            touched = true
        end
    end

    -- Motorized spec: typical place for per-hour operating costs
    local specMotor = vehicle.spec_motorized
    if specMotor ~= nil and type(specMotor) == "table" then
        if type(specMotor.pricePerHour) == "number" and specMotor.pricePerHour ~= 0 then
            zocLog("Zeroing vehicle.spec_motorized.pricePerHour from",
                specMotor.pricePerHour, "to 0")
            specMotor.pricePerHour = 0
            touched = true
        end
        if type(specMotor.operatingCost) == "number" and specMotor.operatingCost ~= 0 then
            zocLog("Zeroing vehicle.spec_motorized.operatingCost from",
                specMotor.operatingCost, "to 0")
            specMotor.operatingCost = 0
            touched = true
        end
    end

    -- Custom maintenance table on vehicle
    local m = vehicle.maintenance
    if m ~= nil and type(m) == "table" then
        local v = m.costPerDay
        if type(v) == "number" and v ~= 0 then
            zocLog("Zeroing vehicle.maintenance.costPerDay from", v, "to 0")
            m.costPerDay = 0
            touched = true
        end
    end

    if touched then
        zocLog("Finished zeroing vehicle costs for", tostring(vehicle))
    end
end

----------------------------------------------------------------------
-- Money hook: block specific money types at balance level
----------------------------------------------------------------------
function ZOC.patchMoney()
    if ZOC._moneyPatched then return end
    if Farm == nil or Farm.changeBalance == nil then
        zocLog("Farm.changeBalance not found, cannot patch money hook")
        return
    end

    local oldChangeBalance = Farm.changeBalance

    Farm.changeBalance = function(farm, amount, moneyType, ...)
        if amount < 0 and isBlockedMoneyType(moneyType) then
            zocLog("Blocking negative money change", amount,
                "for moneyType", tostring(moneyType))
            amount = 0
        end

        return oldChangeBalance(farm, amount, moneyType, ...)
    end

    ZOC._moneyPatched = true
    zocLog("Money hook on Farm.changeBalance installed")
end

----------------------------------------------------------------------
-- HUD hook: suppress money popups for blocked types
-- via g_currentMission.hud:addMoneyChange(...)
--
-- Docs example (fuel purchase) shows:
--   g_currentMission:addMoney(-price, farmId, "purchaseFuel")
--   g_currentMission:addMoneyChange(-price, farmId, self.moneyChangeId)
-- which is ultimately fed into the HUD money change overlay. :contentReference[oaicite:1]{index=1}
----------------------------------------------------------------------
function ZOC.patchNotifications()
    if ZOC._notifPatched then return end

    local mission = g_currentMission
    if mission == nil or mission.hud == nil then
        zocLog("HUD not ready yet, cannot patch HUD notifications")
        return
    end

    local hud = mission.hud
    if hud.addMoneyChange == nil then
        zocLog("HUD.addMoneyChange not found, cannot patch HUD notifications")
        return
    end

    local oldAddMoneyChange = hud.addMoneyChange

    -- NOTE: FS25 signature here is empirically:
    --   hud:addMoneyChange(moneyType, amount, ...)
    -- based on your logs + behavior. If GIANTS changes it,
    -- you can flip argument order easily.
    hud.addMoneyChange = function(self, moneyType, amount, ...)
        if ZOC.debug then
            zocLog("HUD.addMoneyChange called moneyType=", tostring(moneyType),
                   " amount=", tostring(amount))
        end

        if type(amount) == "number" and amount < 0 and isBlockedMoneyType(moneyType) then
            zocLog("Suppressing HUD money popup amount=", amount,
                   " for moneyType=", tostring(moneyType))
            amount = 0
        end

        return oldAddMoneyChange(self, moneyType, amount, ...)
    end

    ZOC._notifPatched = true
    zocLog("HUD money change hook installed on HUD.addMoneyChange")
end

----------------------------------------------------------------------
-- Sweep all production points/lines, placeables, and vehicles
----------------------------------------------------------------------
function ZOC.updateAll()
    local mission = g_currentMission
    if mission == nil then
        zocLog("updateAll: g_currentMission is nil")
        return
    end

    zocLog("Running global ZeroOperatingCosts updateAll sweep")

    --------------------------------------------------------
    -- 1. Production chain manager (production points)
    --------------------------------------------------------
    if ZOC.CONFIG.sweepProductions then
        local pcm = mission.productionChainManager
        if pcm ~= nil and pcm.productionPoints ~= nil then
            for _, productionPoint in pairs(pcm.productionPoints) do
                -- Zero upkeep on the production point object
                ZOC.zeroUpkeepFields(productionPoint)

                -- Zero costs on each production line
                local productions = productionPoint.productions
                if productions ~= nil then
                    for _, prod in pairs(productions) do
                        ZOC.zeroProductionCosts(prod)
                    end
                end
            end
        end
    end

    --------------------------------------------------------
    -- 2. All placeables (buildings, silos, sheds, etc.)
    --------------------------------------------------------
    if ZOC.CONFIG.sweepPlaceables then
        local placeables = mission.placeableSystem
            and mission.placeableSystem.placeables
            or mission.placeables

        if placeables ~= nil then
            for _, placeable in ipairs(placeables) do
                ZOC.zeroUpkeepFields(placeable)

                -- If it has an attached productionPoint, clean that too
                local pp = placeable.productionPoint
                if pp ~= nil then
                    ZOC.zeroUpkeepFields(pp)
                    local productions = pp.productions
                    if productions ~= nil then
                        for _, prod in pairs(productions) do
                            ZOC.zeroProductionCosts(prod)
                        end
                    end
                end
            end
        end
    end

    --------------------------------------------------------
    -- 3. All vehicles / equipment
    --------------------------------------------------------
    if ZOC.CONFIG.sweepVehicles then
        local vehicles = mission.vehicles
        if vehicles ~= nil then
            for _, vehicle in ipairs(vehicles) do
                ZOC.zeroVehicleCosts(vehicle)
            end
        end
    end
end

----------------------------------------------------------------------
-- Event Hooks
----------------------------------------------------------------------
function ZOC:loadMap(mapName)
    zocLog("loadMap called for map:", tostring(mapName))
    self.timeAcc = 0

    -- Install hooks once
    ZOC.patchMoney()
    ZOC.patchNotifications()

    -- Initial cleanup pass
    ZOC.updateAll()
end

function ZOC:update(dt)
    -- Ensure hooks are installed even if loadMap timing was weird
    ZOC.patchMoney()
    ZOC.patchNotifications()

    self.timeAcc = (self.timeAcc or 0) + dt
    if self.timeAcc > (ZOC.CONFIG.scanIntervalMs or 10000) then
        self.timeAcc = 0
        ZOC.updateAll()
    end
end

addModEventListener(ZeroOperatingCosts)

zocLog("ZeroOperatingCosts script fully initialized")
