-- Author: U_BMP
-- Group: https://vk.com/biomodprod_utilit_fs
-- Date: 12.12.2025


ParticleControl = {}
ParticleControl.name = g_currentModName
ParticleControl.path = g_currentModDirectory

ParticleControl.SETTINGS = {}
ParticleControl.CONTROLS = {}
ParticleControl.initialised = false
ParticleControl.menuInstalled = false

addModEventListener(ParticleControl)

ParticleControl.menuItems = {
    "pc_enableWheels",
    "pc_dustMode",
    "pc_wetnessThreshold",
    "pc_wheelEmitMult",
    "pc_enablePersistentDust",
}

ParticleControl.pc_enableWheels      = true
ParticleControl.pc_dustMode          = 1
ParticleControl.pc_wetnessThreshold  = 0.20
ParticleControl.pc_wheelEmitMult     = 2.80
ParticleControl.pc_wheelSizeMult     = 1.45
ParticleControl.pc_wheelSpeedMult    = 1.05
ParticleControl.pc_enablePersistentDust = true
ParticleControl.pc_wheelMinEmit         = 0.15
ParticleControl.pc_wheelGravityMult     = 1.00
ParticleControl.pc_wheelDragMult        = 1.00

local function clamp(x, a, b)
    if x < a then return a end
    if x > b then return b end
    return x
end

ParticleControl.__pcGroundParticlesForced = false
ParticleControl.__pcGroundParticlesDefault = nil
ParticleControl.__pcGroundParticlesDefaultMeta = nil

local function pcCaptureGroundParticlesDefaults()
    if WheelEffects == nil or WheelEffects.GROUND_PARTICLES == nil then
        return
    end
    if ParticleControl.__pcGroundParticlesDefault ~= nil then
        return
    end
    local gp = WheelEffects.GROUND_PARTICLES
    ParticleControl.__pcGroundParticlesDefault = {}
    for k, v in pairs(gp) do
        ParticleControl.__pcGroundParticlesDefault[k] = v
    end
    ParticleControl.__pcGroundParticlesDefaultMeta = getmetatable(gp)
end

local function pcRestoreGroundParticlesDefaults()
    if WheelEffects == nil or WheelEffects.GROUND_PARTICLES == nil then
        return
    end
    if ParticleControl.__pcGroundParticlesDefault == nil then
        return
    end

    local gp = WheelEffects.GROUND_PARTICLES

    setmetatable(gp, ParticleControl.__pcGroundParticlesDefaultMeta)

    for k, v in pairs(ParticleControl.__pcGroundParticlesDefault) do
        gp[k] = v
    end

    ParticleControl.__pcGroundParticlesForced = false
end

local function pcApplyGroundParticlesMode()
    if WheelEffects == nil or WheelEffects.GROUND_PARTICLES == nil then
        return
    end

    pcCaptureGroundParticlesDefaults()

    local gp = WheelEffects.GROUND_PARTICLES
    local dustMode = ParticleControl.pc_dustMode or 1

    if dustMode == 2 then
        for k, _ in pairs(gp) do
            gp[k] = true
        end
        setmetatable(gp, { __index = function() return true end })
        ParticleControl.__pcGroundParticlesForced = true
    else
        if ParticleControl.__pcGroundParticlesForced then
            pcRestoreGroundParticlesDefaults()
        end
    end
end

function ParticleControl:getSettingsFilePath()
    return Utils.getFilename("modSettings/ParticleControl.xml", getUserProfileAppPath())
end

function ParticleControl:setValue(id, value)
    ParticleControl[id] = value
end

function ParticleControl:getValue(id)
    return ParticleControl[id]
end

function ParticleControl:getStateIndex(id, value)
    value = value ~= nil and value or ParticleControl:getValue(id)
    local setting = ParticleControl.SETTINGS[id]
    if not setting then
        return 1
    end

    local values = setting.values
    if type(value) == "number" then
        local index = setting.default
        local bestDiff = math.huge
        for i, v in ipairs(values) do
            local d = math.abs(v - value)
            if d < bestDiff then
                bestDiff = d
                index = i
            end
        end
        return index
    else
        for i, v in ipairs(values) do
            if v == value then
                return i
            end
        end
        return setting.default
    end
end

function ParticleControl:writeSettings()
    local key = "particleControlSettings"
    local filePath = ParticleControl:getSettingsFilePath()
    local xmlFile = createXMLFile("settings", filePath, key)

    if xmlFile ~= 0 then
        for _, id in ipairs(ParticleControl.menuItems) do
            local value = ParticleControl:getValue(id)
            local xmlKey = key .. "." .. id .. "#value"

            if type(value) == "boolean" then
                setXMLBool(xmlFile, xmlKey, value)
            elseif type(value) == "number" then
                setXMLFloat(xmlFile, xmlKey, value)
            else
                setXMLString(xmlFile, xmlKey, tostring(value))
            end
        end
        saveXMLFile(xmlFile)
        delete(xmlFile)
    else
        print("ParticleControl(Wheels): Failed to create XML for saving settings")
    end
end

function ParticleControl:readSettings()
    local filePath = ParticleControl:getSettingsFilePath()
    if not fileExists(filePath) then
        ParticleControl:writeSettings()
        return
    end

    local xmlFile = loadXMLFile("particleControlSettings", filePath)
    if xmlFile == 0 then
        ParticleControl:writeSettings()
        return
    end

    local baseKey = "particleControlSettings"
    for _, id in ipairs(ParticleControl.menuItems) do
        local xmlKey = baseKey .. "." .. id .. "#value"
        if hasXMLProperty(xmlFile, xmlKey) then
            if type(ParticleControl[id]) == "boolean" then
                local v = getXMLBool(xmlFile, xmlKey)
                if v ~= nil then ParticleControl:setValue(id, v) end
            else
                local v = getXMLFloat(xmlFile, xmlKey)
                if v ~= nil then ParticleControl:setValue(id, v) end
            end
        end
    end
    delete(xmlFile)
end

local function pcApplyGravityAndDragIfPossible(ps)
    if ps == nil then return end
    local gMult = ParticleControl.pc_wheelGravityMult or 1.0
    local dMult = ParticleControl.pc_wheelDragMult or 1.0
    if gMult ~= 1.0 then
        if ParticleUtil.getParticleSystemGravity ~= nil and ParticleUtil.setParticleSystemGravity ~= nil then
            local gx, gy, gz = ParticleUtil.getParticleSystemGravity(ps)
            if gx ~= nil and gy ~= nil and gz ~= nil then
                ParticleUtil.setParticleSystemGravity(ps, gx * gMult, gy * gMult, gz * gMult)
            end
        elseif ParticleUtil.getParticleSystemGravityY ~= nil and ParticleUtil.setParticleSystemGravityY ~= nil then
            local gy = ParticleUtil.getParticleSystemGravityY(ps)
            if gy ~= nil then
                ParticleUtil.setParticleSystemGravityY(ps, gy * gMult)
            end
        end
    end
    if dMult ~= 1.0 then
        if ParticleUtil.getParticleSystemDrag ~= nil and ParticleUtil.setParticleSystemDrag ~= nil then
            local d = ParticleUtil.getParticleSystemDrag(ps)
            if d ~= nil then
                ParticleUtil.setParticleSystemDrag(ps, d * dMult)
            end
        elseif ParticleUtil.getParticleSystemDamping ~= nil and ParticleUtil.setParticleSystemDamping ~= nil then
            local d = ParticleUtil.getParticleSystemDamping(ps)
            if d ~= nil then
                ParticleUtil.setParticleSystemDamping(ps, d * dMult)
            end
        end
    end
end

local function pcRestoreWheelParticleSystemDefaults(ps)
    if ps == nil then return end

    if ps.__pcOrigSpriteScaleX ~= nil and ParticleUtil.setParticleSystemSpriteScaleX ~= nil then
        ParticleUtil.setParticleSystemSpriteScaleX(ps, ps.__pcOrigSpriteScaleX)
    end
    if ps.__pcOrigSpriteScaleY ~= nil and ParticleUtil.setParticleSystemSpriteScaleY ~= nil then
        ParticleUtil.setParticleSystemSpriteScaleY(ps, ps.__pcOrigSpriteScaleY)
    end
    if ps.__pcOrigSpriteScaleXGain ~= nil and ParticleUtil.setParticleSystemSpriteScaleXGain ~= nil then
        ParticleUtil.setParticleSystemSpriteScaleXGain(ps, ps.__pcOrigSpriteScaleXGain)
    end
    if ps.__pcOrigSpriteScaleYGain ~= nil and ParticleUtil.setParticleSystemSpriteScaleYGain ~= nil then
        ParticleUtil.setParticleSystemSpriteScaleYGain(ps, ps.__pcOrigSpriteScaleYGain)
    end

    if ps.__pcOrigParticleSpeed ~= nil and ParticleUtil.setParticleSystemSpeed ~= nil then
        ps.particleSpeed = ps.__pcOrigParticleSpeed
        ParticleUtil.setParticleSystemSpeed(ps, ps.__pcOrigParticleSpeed)
    end
    if ps.__pcOrigParticleRandomSpeed ~= nil and ParticleUtil.setParticleSystemSpeedRandom ~= nil then
        ps.particleRandomSpeed = ps.__pcOrigParticleRandomSpeed
        ParticleUtil.setParticleSystemSpeedRandom(ps, ps.__pcOrigParticleRandomSpeed)
    end
end

function ParticleControl.installHooks()
    if ParticleControl._hooksInstalled then
        return
    end
    ParticleControl._hooksInstalled = true

    if WheelEffects == nil then
        print("ParticleControl(Wheels): WheelEffects not found (unexpected load order?)")
        return
    end

    pcCaptureGroundParticlesDefaults()

    WheelEffects.onWheelParticleSystemI3DLoaded = Utils.overwrittenFunction(
        WheelEffects.onWheelParticleSystemI3DLoaded,
        function(self, superFunc, i3dNode, failedReason, args)
            if superFunc ~= nil then
                superFunc(self, i3dNode, failedReason, args)
            end

            if not ParticleControl.pc_enableWheels then
                self.__pcRestoredToDefault = true
                return
            end

            local ps = self.driveGroundParticleSystems and self.driveGroundParticleSystems[#self.driveGroundParticleSystems] or nil
            if ps == nil then
                return
            end

            if ps.__pcApplied then
                return
            end
            ps.__pcApplied = true

            ps.__pcOrigSpriteScaleX = ParticleUtil.getParticleSystemSpriteScaleX(ps)
            ps.__pcOrigSpriteScaleY = ParticleUtil.getParticleSystemSpriteScaleY(ps)
            ps.__pcOrigSpriteScaleXGain = ParticleUtil.getParticleSystemSpriteScaleXGain(ps)
            ps.__pcOrigSpriteScaleYGain = ParticleUtil.getParticleSystemSpriteScaleYGain(ps)
            ps.__pcOrigParticleSpeed = ps.particleSpeed
            ps.__pcOrigParticleRandomSpeed = ps.particleRandomSpeed

            local sizeMult  = ParticleControl.pc_wheelSizeMult or 1.0
            local speedMult = ParticleControl.pc_wheelSpeedMult or 1.0

            if ps.__pcOrigSpriteScaleX ~= nil then
                ParticleUtil.setParticleSystemSpriteScaleX(ps, ps.__pcOrigSpriteScaleX * sizeMult)
            end
            if ps.__pcOrigSpriteScaleY ~= nil then
                ParticleUtil.setParticleSystemSpriteScaleY(ps, ps.__pcOrigSpriteScaleY * sizeMult)
            end
            if ps.__pcOrigSpriteScaleXGain ~= nil then
                ParticleUtil.setParticleSystemSpriteScaleXGain(ps, ps.__pcOrigSpriteScaleXGain * sizeMult)
            end
            if ps.__pcOrigSpriteScaleYGain ~= nil then
                ParticleUtil.setParticleSystemSpriteScaleYGain(ps, ps.__pcOrigSpriteScaleYGain * sizeMult)
            end

            if ps.__pcOrigParticleSpeed ~= nil then
                ps.particleSpeed = ps.__pcOrigParticleSpeed * speedMult
                ParticleUtil.setParticleSystemSpeed(ps, ps.particleSpeed)
            end
            if ps.__pcOrigParticleRandomSpeed ~= nil then
                ps.particleRandomSpeed = ps.__pcOrigParticleRandomSpeed * speedMult
                ParticleUtil.setParticleSystemSpeedRandom(ps, ps.particleRandomSpeed)
            end

            pcApplyGravityAndDragIfPossible(ps)

            self.__pcRestoredToDefault = false
        end
    )

    WheelEffects.updateTick = Utils.overwrittenFunction(
        WheelEffects.updateTick,
        function(self, superFunc, dt, groundWetness, currentUpdateDistance)
            if superFunc ~= nil then
                superFunc(self, dt, groundWetness, currentUpdateDistance)
            end

            if not ParticleControl.pc_enableWheels then
                pcRestoreGroundParticlesDefaults()

                if self.__pcRestoredToDefault ~= true then
                    if self.driveGroundParticleSystems ~= nil then
                        for _, ps in ipairs(self.driveGroundParticleSystems) do
                            pcRestoreWheelParticleSystemDefaults(ps)
                        end
                    end

                    self.__pcRestoredToDefault = true
                end
                return
            end

            self.__pcRestoredToDefault = false

            pcApplyGroundParticlesMode()

            local emitMult = ParticleControl.pc_wheelEmitMult or 1.0
            local persistent = ParticleControl.pc_enablePersistentDust == true

            local minEmit = ParticleControl.pc_wheelMinEmit or 0.0
            minEmit = clamp(minEmit, 0.0, 1.0)

            if self.driveGroundParticleSystems ~= nil then
                for _, ps in ipairs(self.driveGroundParticleSystems) do
                    local wheelPhysics = self.wheel and self.wheel.physics
                    if wheelPhysics ~= nil and ps ~= nil and ps.state ~= nil then
                        local netInfo = wheelPhysics.netInfo
                        local speedForScale = (netInfo and netInfo.lastSpeedSmoothed) or 0
                        local slip = (netInfo and netInfo.slip) or 0
                        local slipScale = 1 + slip

                        local baseScale
                        if WheelEffects.PARTICLE_SYSTEM_STATES ~= nil
                           and ps.state == WheelEffects.PARTICLE_SYSTEM_STATES.WHEEL_DUST then
                            baseScale = self:getDriveGroundParticleSystemsScale(ps, self.vehicle.lastSpeedSmoothed)
                        else
                            baseScale = self:getDriveGroundParticleSystemsScale(ps, speedForScale) * slipScale
                        end

                        local desiredScale = 0

                        if baseScale > 0 then
                            local v140 = 13.88888888888889 / wheelPhysics.radiusOriginal
                            local xDrive = (netInfo and netInfo.xDriveSpeed) or 0
                            local v141 = baseScale * (xDrive / v140) * ps.sizeScale
                            desiredScale = v141 * emitMult
                        end

                        local minS = self.minScale or 0.1
                        local maxS = (self.maxScale or 1.0) * emitMult

                        if persistent and minEmit > 0 then
                            local floor = clamp(minEmit, minS, maxS)
                            desiredScale = math.max(desiredScale, floor)
                        else
                            if desiredScale > 0 then
                                desiredScale = clamp(desiredScale, minS, maxS)
                            end
                        end

                        ParticleUtil.setEmitCountScale(ps, desiredScale)
                    end
                end
            end
        end
    )
end

local function buildPercentValues(minPercent, maxPercent, stepPercent, format)
    local values = {}
    local strings = {}
    local idx = 1
    for p = minPercent, maxPercent, stepPercent do
        values[idx] = p / 100
        strings[idx] = string.format(format or "%.0f%%", p)
        idx = idx + 1
    end
    return values, strings
end

local multValues, multStrings = buildPercentValues(0, 550, 1, "%.0f%%")
local wetValues, wetStrings   = buildPercentValues(0, 100, 1, "%.0f%%")

ParticleControl.SETTINGS.pc_enableWheels = {
    default = 2,
    values = { false, true },
    strings = { g_i18n:getText("ui_off"), g_i18n:getText("ui_on") }
}

ParticleControl.SETTINGS.pc_dustMode = {
    default = 1,
    values = { 1, 2 },
    strings = { g_i18n:getText("pc_dustMode_engine"), g_i18n:getText("pc_dustMode_everywhere") }
}

ParticleControl.SETTINGS.pc_wetnessThreshold = {
    default = 21,
    values = wetValues,
    strings = wetStrings
}

ParticleControl.SETTINGS.pc_wheelEmitMult = {
    default = 57,
    values = multValues,
    strings = multStrings
}

ParticleControl.SETTINGS.pc_enablePersistentDust = {
    default = 1,
    values = { false, true },
    strings = { g_i18n:getText("ui_off"), g_i18n:getText("ui_on") }
}

function ParticleControl:onMenuOptionChanged(state, menuOption)
    local id = menuOption.id
    local value = ParticleControl.SETTINGS[id].values[state]
    ParticleControl:setValue(id, value)
    ParticleControl:writeSettings()

    if id == "pc_enableWheels" then
        if ParticleControl.pc_enableWheels then
            pcApplyGroundParticlesMode()
        else
            pcRestoreGroundParticlesDefaults()
        end
    end

    if id == "pc_dustMode" then
        if ParticleControl.pc_enableWheels then
            pcApplyGroundParticlesMode()
        else
            pcRestoreGroundParticlesDefaults()
        end
    end
end

function ParticleControl:addMenuOption(id, settingsPage, settingsLayout)
    local original = (#ParticleControl.SETTINGS[id].values == 2)
        and settingsPage.checkWoodHarvesterAutoCutBox
        or settingsPage.multiVolumeVoiceBox

    local options = ParticleControl.SETTINGS[id].strings

    local menuOptionBox = original:clone(settingsLayout)
    menuOptionBox.id = id .. "Box"

    local menuOption = menuOptionBox.elements[1]
    menuOption.id = id
    menuOption.target = ParticleControl
    menuOption:setCallback("onClickCallback", "onMenuOptionChanged")
    menuOption:setDisabled(false)

    local toolTip = menuOption.elements[1]
    toolTip:setText(g_i18n:getText("tooltip_particlecontrol_" .. id))

    local setting = menuOptionBox.elements[2]
    setting:setText(g_i18n:getText("setting_particlecontrol_" .. id))

    menuOption:setTexts({ unpack(options) })
    menuOption:setState(ParticleControl:getStateIndex(id))

    ParticleControl.CONTROLS[id] = menuOption

    local function updateFocusIds(element)
        if element == nil then return end
        element.focusId = FocusManager:serveAutoFocusId()
        for _, child in pairs(element.elements) do
            updateFocusIds(child)
        end
    end
    updateFocusIds(menuOptionBox)

    table.insert(settingsPage.controlsList, menuOptionBox)
end

function ParticleControl:installMenuIfPossible()
    if ParticleControl.menuInstalled then
        return
    end

    local inGameMenu = g_gui and g_gui.screenControllers and g_gui.screenControllers[InGameMenu]
    if inGameMenu == nil or inGameMenu.pageSettings == nil then
        return
    end

    local settingsPage = inGameMenu.pageSettings
    local settingsLayout = settingsPage.generalSettingsLayout
    if settingsLayout == nil then
        return
    end

    local sectionTitle
    for _, elem in ipairs(settingsLayout.elements) do
        if elem.name == "sectionHeader" then
            sectionTitle = elem:clone(settingsLayout)
            break
        end
    end

    if sectionTitle ~= nil then
        sectionTitle:setText(g_i18n:getText("menu_ParticleControl_WHEELS_TITLE"))
    else
        local title = TextElement.new()
        title:applyProfile("fs25_settingsSectionHeader", true)
        title:setText(g_i18n:getText("menu_ParticleControl_WHEELS_TITLE"))
        title.name = "sectionHeader"
        settingsLayout:addElement(title)
        sectionTitle = title
    end

    sectionTitle.focusId = FocusManager:serveAutoFocusId()
    table.insert(settingsPage.controlsList, sectionTitle)
    ParticleControl.CONTROLS[sectionTitle.name] = sectionTitle

    for _, id in ipairs(ParticleControl.menuItems) do
        ParticleControl:addMenuOption(id, settingsPage, settingsLayout)
    end

    settingsLayout:invalidateLayout()

    FocusManager.setGui = Utils.appendedFunction(FocusManager.setGui, function(_, gui)
        if gui == "ingameMenuSettings" then
            for _, control in pairs(ParticleControl.CONTROLS) do
                if control.focusId ~= nil and not FocusManager.currentFocusData.idToElementMapping[control.focusId] then
                    if not FocusManager:loadElementFromCustomValues(control, nil, nil, false, false) then
                        Logging.warning("ParticleControl(Wheels): Could not register control %s with focus manager", control.id or control.name or control.focusId)
                    end
                end
            end
            local sp = g_gui.screenControllers[InGameMenu].pageSettings
            sp.generalSettingsLayout:invalidateLayout()
        end
    end)

    InGameMenuSettingsFrame.onFrameOpen = Utils.appendedFunction(InGameMenuSettingsFrame.onFrameOpen, function()
        local isAdmin = g_currentMission:getIsServer() or g_currentMission.isMasterUser
        for _, id in ipairs(ParticleControl.menuItems) do
            local menuOption = ParticleControl.CONTROLS[id]
            if menuOption ~= nil then
                menuOption:setState(ParticleControl:getStateIndex(id))
                menuOption:setDisabled(not isAdmin)
            end
        end
    end)

    ParticleControl.menuInstalled = true
end

function ParticleControl:loadMap()
    ParticleControl:readSettings()

    addConsoleCommand("gsParticleControl", "Print ParticleControl wheel values", "consoleCommandParticleControl", ParticleControl)

    ParticleControl.installHooks()
    ParticleControl.initialised = true

    if ParticleControl.pc_enableWheels then
        pcApplyGroundParticlesMode()
    else
        pcRestoreGroundParticlesDefaults()
    end

    if g_currentMission ~= nil then
        FSBaseMission.registerToLoadOnMapFinished(g_currentMission, ParticleControl)
    end
end

function ParticleControl:onLoadMapFinished()
    ParticleControl:readSettings()

    if ParticleControl.pc_enableWheels then
        pcApplyGroundParticlesMode()
    else
        pcRestoreGroundParticlesDefaults()
    end

    ParticleControl:installMenuIfPossible()
end

function ParticleControl:update(dt)
    if ParticleControl.initialised and not ParticleControl.menuInstalled then
        ParticleControl:installMenuIfPossible()
    end
end

function ParticleControl:consoleCommandParticleControl()
    local out = {}
    for _, id in ipairs(ParticleControl.menuItems) do
        out[#out + 1] = string.format("%s=%s", id, tostring(ParticleControl:getValue(id)))
    end

    out[#out + 1] = string.format("INTERNAL(pc_wheelSizeMult)=%s", tostring(ParticleControl.pc_wheelSizeMult))
    out[#out + 1] = string.format("INTERNAL(pc_wheelSpeedMult)=%s", tostring(ParticleControl.pc_wheelSpeedMult))
    out[#out + 1] = string.format("INTERNAL(pc_wheelMinEmit)=%s", tostring(ParticleControl.pc_wheelMinEmit))
    out[#out + 1] = string.format("INTERNAL(pc_wheelGravityMult)=%s", tostring(ParticleControl.pc_wheelGravityMult))
    out[#out + 1] = string.format("INTERNAL(pc_wheelDragMult)=%s", tostring(ParticleControl.pc_wheelDragMult))

    return "ParticleControl(Wheels): " .. table.concat(out, " | ")
end
