-- ============================================================= --
-- SUN PRESETS (FS25)
-- Правый Ctrl + ←/→ переключают пресеты в игре.
-- Появляется короткий HUD-тост с названием пресета.
-- Мягкое применение без "миганий ночи".
-- Сохранение/загрузка как в FogControl: modSettings/SunPresets.xml
-- дата: 06.11.2025
-- ============================================================= --
SunPresets = {}
SunPresets.name = g_currentModName
SunPresets.path = g_currentModDirectory

SunPresets.CONTROLS = {}
SunPresets.ready = false       -- станет true после Lighting:updateCurves()
SunPresets.selected = 1        -- индекс текущего пресета (1..#presets)

-- HUD-тост
SunPresets.toastUntil = nil    -- время (ms), до которого рисуем тост
SunPresets.toastText  = nil    -- локализованное имя пресета
SunPresets.toastSec   = 2500   -- длительность показа, ms

addModEventListener(SunPresets)

------------------------------------------------------------------
-- УТИЛИТЫ
------------------------------------------------------------------
local function safeCall(funcName, ...)
    local f = _G[funcName]
    if type(f) == "function" then
        return pcall(f, ...)
    end
    return false
end

local function clamp(v, lo, hi)
    if v < lo then return lo end
    if v > hi then return hi end
    return v
end

-- === пути настроек в стиле FogControl (+ поддержка старого пути) ===
local function _getSettingsPaths()
    -- Новый путь (как в FogControl)
    local newPath = Utils.getFilename("modSettings/SunPresets.xml", getUserProfileAppPath())
    -- Легаси путь (из ранних версий мода)
    local legacyPath = Utils.getFilename(string.format("modSettings/%s_SunPresets.xml", g_currentModName), getUserProfileAppPath())
    return newPath, legacyPath
end

------------------------------------------------------------------
-- ПРЕСЕТЫ (40 шт.)
-- bMul/sMul — множители яркости/размера солнца
-- r/g/b — множители цвета солнца
-- bloomMag/bloomThr — пост-эффект bloom
-- toneSlope/toneToe — тонмаппинг
-- cg{} — мягкий color grading поверх карты
------------------------------------------------------------------
SunPresets.presets = {
    { key="sp_preset_default", -- 1
      bMul=1.0, sMul=1.0, r=1.0, g=1.0, b=1.0,
      bloomMag=0.50, bloomThr=2.00, toneSlope=1.00, toneToe=0.40,
      cg={ global={1,1,1,1}, shadows={1,1,1,1}, mids={1,1,1,1}, highs={1,1,1,1}, thr={0.04,0.85} } },

    { key="sp_preset_cinematicWarm", -- 2
      bMul=1.2, sMul=1.1, r=1.2, g=1.05, b=0.95,
      bloomMag=0.65, bloomThr=1.80, toneSlope=0.95, toneToe=0.36,
      cg={ global={1.05,1.08,1.0,1.05}, shadows={1.0,1.05,1.0,1.0}, mids={1.05,1.05,1.0,1.02}, highs={0.98,1.02,1.0,1.0}, thr={0.05,0.82} } },

    { key="sp_preset_cinematicCool", -- 3
      bMul=1.1, sMul=1.0, r=0.95, g=1.05, b=1.12,
      bloomMag=0.55, bloomThr=2.10, toneSlope=1.05, toneToe=0.38,
      cg={ global={0.98,1.06,1.02,1.0}, shadows={1.02,1.05,1.02,1.0}, mids={0.98,1.04,1.01,1.0}, highs={1.02,1.02,1.0,1.0}, thr={0.05,0.86} } },

    { key="sp_preset_blueHour", -- 4
      bMul=0.9, sMul=1.0, r=0.92, g=1.0, b=1.15,
      bloomMag=0.40, bloomThr=1.80, toneSlope=0.90, toneToe=0.32,
      cg={ global={0.98,1.02,1.02,1.0}, shadows={1.05,1.05,1.0,1.0}, mids={0.98,1.05,1.02,1.0}, highs={1.02,1.00,1.0,1.0}, thr={0.07,0.80} } },

    { key="sp_preset_goldenHour", -- 5
      bMul=1.35, sMul=1.15, r=1.25, g=1.10, b=0.90,
      bloomMag=0.80, bloomThr=1.60, toneSlope=0.92, toneToe=0.34,
      cg={ global={1.08,1.08,1.0,1.05}, shadows={1.00,1.03,1.0,1.0}, mids={1.06,1.06,1.0,1.02}, highs={1.00,1.02,1.0,1.0}, thr={0.05,0.83} } },

    { key="sp_preset_overcastSoft", -- 6
      bMul=0.85, sMul=1.0, r=1.0, g=1.0, b=1.0,
      bloomMag=0.30, bloomThr=2.50, toneSlope=1.10, toneToe=0.45,
      cg={ global={0.96,1.00,1.02,0.98}, shadows={1.02,1.02,1.0,1.0}, mids={0.98,1.00,1.0,1.0}, highs={0.98,0.98,1.0,0.98}, thr={0.06,0.88} } },

    { key="sp_preset_highContrastSun", -- 7
      bMul=1.6, sMul=1.2, r=1.1, g=1.1, b=1.05,
      bloomMag=0.70, bloomThr=2.20, toneSlope=1.25, toneToe=0.38,
      cg={ global={1.00,1.12,1.0,1.0}, shadows={1.00,1.08,1.0,1.0}, mids={1.00,1.10,1.0,1.0}, highs={1.00,1.06,1.0,1.0}, thr={0.04,0.90} } },

    { key="sp_preset_softFilm", -- 8
      bMul=1.05, sMul=1.05, r=1.04, g=1.02, b=0.98,
      bloomMag=0.60, bloomThr=1.90, toneSlope=0.88, toneToe=0.42,
      cg={ global={1.02,1.04,1.02,1.02}, shadows={1.00,1.02,1.0,1.0}, mids={1.02,1.03,1.0,1.0}, highs={0.98,1.00,1.0,1.0}, thr={0.05,0.84} } },

    { key="sp_preset_desaturatedNoir", -- 9
      bMul=1.2, sMul=1.0, r=1.0, g=1.0, b=1.0,
      bloomMag=0.35, bloomThr=2.80, toneSlope=1.30, toneToe=0.50,
      cg={ global={0.30,1.10,1.00,1.00}, shadows={0.30,1.10,1.00,1.00}, mids={0.30,1.10,1.00,1.00}, highs={0.30,1.05,1.00,1.00}, thr={0.05,0.88} } },

    { key="sp_preset_vibrant", -- 10
      bMul=1.25, sMul=1.05, r=1.08, g=1.08, b=1.08,
      bloomMag=0.55, bloomThr=1.90, toneSlope=1.00, toneToe=0.38,
      cg={ global={1.15,1.06,1.00,1.05}, shadows={1.08,1.04,1.00,1.00}, mids={1.10,1.05,1.00,1.02}, highs={1.05,1.02,1.00,1.00}, thr={0.04,0.86} } },

    { key="sp_preset_fadeVintage", -- 11
      bMul=1.0, sMul=1.0, r=1.08, g=1.02, b=0.94,
      bloomMag=0.65, bloomThr=1.70, toneSlope=0.85, toneToe=0.46,
      cg={ global={0.92,1.02,1.05,1.02}, shadows={0.95,1.02,1.05,1.00}, mids={0.92,1.02,1.05,1.02}, highs={0.90,1.00,1.05,1.02}, thr={0.06,0.80} } },

    { key="sp_preset_warmDesert", -- 12
      bMul=1.5, sMul=1.2, r=1.18, g=1.08, b=0.90,
      bloomMag=0.70, bloomThr=2.10, toneSlope=1.10, toneToe=0.36,
      cg={ global={1.06,1.08,1.00,1.04}, shadows={1.02,1.06,1.00,1.00}, mids={1.06,1.08,1.00,1.02}, highs={1.00,1.02,1.00,1.00}, thr={0.05,0.88} } },

    { key="sp_preset_coldWinter", -- 13
      bMul=1.1, sMul=1.0, r=0.95, g=1.02, b=1.20,
      bloomMag=0.45, bloomThr=2.40, toneSlope=1.05, toneToe=0.42,
      cg={ global={0.95,1.04,1.02,0.98}, shadows={1.02,1.04,1.00,0.98}, mids={0.96,1.04,1.02,0.98}, highs={1.00,1.02,1.00,1.00}, thr={0.06,0.90} } },

    { key="sp_preset_lowKeyNight", -- 14
      bMul=0.7, sMul=1.0, r=0.90, g=0.95, b=1.10,
      bloomMag=0.30, bloomThr=3.00, toneSlope=1.35, toneToe=0.55,
      cg={ global={0.80,1.08,1.02,0.95}, shadows={0.85,1.10,1.02,0.95}, mids={0.82,1.08,1.02,0.95}, highs={0.85,1.05,1.02,0.95}, thr={0.05,0.92} } },

    { key="sp_preset_moonlitBlue", -- 15
      bMul=0.8, sMul=1.0, r=0.90, g=0.98, b=1.18,
      bloomMag=0.35, bloomThr=2.60, toneSlope=1.15, toneToe=0.50,
      cg={ global={0.90,1.04,1.02,0.98}, shadows={0.95,1.06,1.02,0.98}, mids={0.92,1.05,1.02,0.98}, highs={0.92,1.02,1.02,0.98}, thr={0.06,0.90} } },

    { key="sp_preset_hazyMorning", -- 16
      bMul=1.05, sMul=1.1, r=1.05, g=1.05, b=1.0,
      bloomMag=0.75, bloomThr=1.50, toneSlope=0.88, toneToe=0.34,
      cg={ global={1.00,1.00,1.00,1.02}, shadows={1.02,1.00,1.00,1.00}, mids={1.02,1.00,1.00,1.00}, highs={1.00,0.98,1.00,1.00}, thr={0.07,0.82} } },

    { key="sp_preset_clearNoonPunch", -- 17
      bMul=1.7, sMul=1.15, r=1.05, g=1.05, b=1.05,
      bloomMag=0.60, bloomThr=2.40, toneSlope=1.20, toneToe=0.36,
      cg={ global={1.04,1.10,1.00,1.02}, shadows={1.00,1.08,1.00,1.00}, mids={1.02,1.10,1.00,1.02}, highs={1.00,1.06,1.00,1.00}, thr={0.04,0.90} } },

    { key="sp_preset_autumnAmber", -- 18
      bMul=1.3, sMul=1.05, r=1.18, g=1.06, b=0.92,
      bloomMag=0.65, bloomThr=1.85, toneSlope=0.95, toneToe=0.38,
      cg={ global={1.08,1.06,1.00,1.04}, shadows={1.04,1.04,1.00,1.00}, mids={1.08,1.05,1.00,1.02}, highs={1.02,1.02,1.00,1.00}, thr={0.05,0.86} } },

    { key="sp_preset_neutralFlat", -- 19
      bMul=1.0, sMul=1.0, r=1.0, g=1.0, b=1.0,
      bloomMag=0.40, bloomThr=2.20, toneSlope=0.95, toneToe=0.48,
      cg={ global={1.00,0.94,1.02,1.00}, shadows={1.00,0.98,1.02,1.00}, mids={1.00,0.96,1.02,1.00}, highs={1.00,0.96,1.02,1.00}, thr={0.05,0.88} } },

    { key="sp_preset_hdrPop", -- 20
      bMul=1.5, sMul=1.2, r=1.08, g=1.08, b=1.08,
      bloomMag=0.85, bloomThr=1.70, toneSlope=1.30, toneToe=0.35,
      cg={ global={1.10,1.12,1.00,1.06}, shadows={1.05,1.10,1.00,1.00}, mids={1.10,1.12,1.00,1.04}, highs={1.05,1.08,1.00,1.02}, thr={0.04,0.86} } },

    -- 21
    { key="sp_preset_duskCopper",
      bMul=1.25, sMul=1.08, r=1.22, g=1.05, b=0.92,
      bloomMag=0.72, bloomThr=1.70, toneSlope=0.96, toneToe=0.37,
      cg={ global={1.06,1.06,1.00,1.04}, shadows={1.00,1.04,1.00,1.00}, mids={1.08,1.06,1.00,1.02}, highs={1.02,1.02,1.00,1.00}, thr={0.05,0.84} } },

    -- 22
    { key="sp_preset_sunrisePeach",
      bMul=1.30, sMul=1.12, r=1.18, g=1.08, b=0.96,
      bloomMag=0.78, bloomThr=1.55, toneSlope=0.90, toneToe=0.33,
      cg={ global={1.04,1.04,1.00,1.03}, shadows={1.02,1.02,1.00,1.00}, mids={1.06,1.04,1.00,1.02}, highs={1.00,1.00,1.00,1.00}, thr={0.06,0.80} } },

    -- 23
    { key="sp_preset_forestGreenTint",
      bMul=1.10, sMul=1.00, r=0.98, g=1.10, b=1.02,
      bloomMag=0.48, bloomThr=2.20, toneSlope=1.06, toneToe=0.40,
      cg={ global={1.02,1.04,1.00,1.00}, shadows={1.04,1.06,1.00,1.00}, mids={1.02,1.05,1.00,1.00}, highs={1.00,1.02,1.00,1.00}, thr={0.05,0.88} } },

    -- 24
    { key="sp_preset_foggyFilm",
      bMul=0.90, sMul=1.06, r=1.02, g=1.02, b=1.04,
      bloomMag=0.85, bloomThr=1.45, toneSlope=0.86, toneToe=0.36,
      cg={ global={0.94,0.98,1.02,1.00}, shadows={0.98,1.00,1.02,1.00}, mids={0.96,0.98,1.02,1.00}, highs={0.94,0.96,1.02,0.98}, thr={0.07,0.82} } },

    -- 25
    { key="sp_preset_crispAlpine",
      bMul=1.35, sMul=1.05, r=0.98, g=1.05, b=1.15,
      bloomMag=0.50, bloomThr=2.30, toneSlope=1.18, toneToe=0.38,
      cg={ global={0.98,1.06,1.02,1.00}, shadows={1.00,1.06,1.02,1.00}, mids={0.98,1.06,1.02,1.00}, highs={1.00,1.04,1.02,1.00}, thr={0.05,0.90} } },

    -- 26
    { key="sp_preset_pastelDawn",
      bMul=1.10, sMul=1.10, r=1.08, g=1.04, b=1.02,
      bloomMag=0.70, bloomThr=1.60, toneSlope=0.92, toneToe=0.35,
      cg={ global={1.04,1.02,1.00,1.02}, shadows={1.02,1.02,1.00,1.00}, mids={1.04,1.02,1.00,1.02}, highs={1.00,1.00,1.00,1.00}, thr={0.06,0.82} } },

    -- 27
    { key="sp_preset_stormDrama",
      bMul=1.40, sMul=1.18, r=1.00, g=1.00, b=1.00,
      bloomMag=0.42, bloomThr=2.80, toneSlope=1.34, toneToe=0.44,
      cg={ global={0.92,1.14,1.00,1.00}, shadows={0.90,1.18,1.00,1.00}, mids={0.92,1.16,1.00,1.00}, highs={0.96,1.10,1.00,1.00}, thr={0.05,0.92} } },

    -- 28
    { key="sp_preset_neonNight",
      bMul=0.80, sMul=1.00, r=1.05, g=0.95, b=1.18,
      bloomMag=0.88, bloomThr=1.75, toneSlope=1.22, toneToe=0.52,
      cg={ global={1.10,1.08,1.00,1.08}, shadows={1.08,1.06,1.00,1.04}, mids={1.12,1.08,1.00,1.06}, highs={1.10,1.06,1.00,1.04}, thr={0.04,0.90} } },

    -- 29
    { key="sp_preset_cyberBlue",
      bMul=1.05, sMul=1.02, r=0.92, g=1.00, b=1.20,
      bloomMag=0.68, bloomThr=1.90, toneSlope=1.10, toneToe=0.46,
      cg={ global={1.06,1.10,1.00,1.04}, shadows={1.04,1.08,1.00,1.02}, mids={1.06,1.10,1.00,1.04}, highs={1.04,1.06,1.00,1.02}, thr={0.05,0.88} } },

    -- 30
    { key="sp_preset_sepiaSun",
      bMul=1.20, sMul=1.10, r=1.18, g=1.04, b=0.90,
      bloomMag=0.75, bloomThr=1.65, toneSlope=0.88, toneToe=0.48,
      cg={ global={0.88,1.04,1.04,1.04}, shadows={0.90,1.04,1.04,1.02}, mids={0.90,1.04,1.04,1.02}, highs={0.88,1.02,1.04,1.02}, thr={0.06,0.84} } },

    -- 31
    { key="sp_preset_winterNoSun",
      bMul=0.75, sMul=1.00, r=0.96, g=1.02, b=1.12,
      bloomMag=0.30, bloomThr=2.90, toneSlope=1.28, toneToe=0.52,
      cg={ global={0.88,1.06,1.02,0.98}, shadows={0.92,1.08,1.02,0.98}, mids={0.90,1.06,1.02,0.98}, highs={0.90,1.04,1.02,0.98}, thr={0.05,0.94} } },

    -- 32
    { key="sp_preset_radiantSpring",
      bMul=1.40, sMul=1.08, r=1.06, g=1.14, b=1.04,
      bloomMag=0.62, bloomThr=1.85, toneSlope=1.02, toneToe=0.36,
      cg={ global={1.10,1.08,1.00,1.06}, shadows={1.06,1.06,1.00,1.02}, mids={1.10,1.08,1.00,1.04}, highs={1.04,1.04,1.00,1.02}, thr={0.05,0.86} } },

    -- 33
    { key="sp_preset_harvestOrange",
      bMul=1.45, sMul=1.12, r=1.22, g=1.10, b=0.92,
      bloomMag=0.72, bloomThr=1.70, toneSlope=0.94, toneToe=0.36,
      cg={ global={1.12,1.08,1.00,1.06}, shadows={1.04,1.06,1.00,1.02}, mids={1.10,1.08,1.00,1.04}, highs={1.02,1.04,1.00,1.02}, thr={0.05,0.84} } },

    -- 34
    { key="sp_preset_silverMoon",
      bMul=0.85, sMul=1.00, r=0.94, g=0.98, b=1.16,
      bloomMag=0.40, bloomThr=2.60, toneSlope=1.20, toneToe=0.50,
      cg={ global={0.96,1.04,1.02,0.98}, shadows={0.98,1.06,1.02,0.98}, mids={0.96,1.04,1.02,0.98}, highs={0.96,1.02,1.02,0.98}, thr={0.06,0.90} } },

    -- 35
    { key="sp_preset_magentaTwilight",
      bMul=1.15, sMul=1.04, r=1.16, g=0.98, b=1.08,
      bloomMag=0.78, bloomThr=1.60, toneSlope=0.92, toneToe=0.34,
      cg={ global={1.06,1.06,1.00,1.04}, shadows={1.04,1.04,1.00,1.02}, mids={1.08,1.06,1.00,1.04}, highs={1.04,1.02,1.00,1.02}, thr={0.05,0.82} } },

    -- 36
    { key="sp_preset_rainSoaked",
      bMul=0.95, sMul=1.00, r=0.98, g=1.02, b=1.06,
      bloomMag=0.50, bloomThr=2.40, toneSlope=1.14, toneToe=0.44,
      cg={ global={0.94,1.06,1.02,1.00}, shadows={0.96,1.08,1.02,1.00}, mids={0.96,1.06,1.02,1.00}, highs={0.98,1.04,1.02,0.98}, thr={0.06,0.90} } },

    -- 37
    { key="sp_preset_desertNoonHard",
      bMul=1.80, sMul=1.22, r=1.08, g=1.06, b=1.00,
      bloomMag=0.52, bloomThr=2.70, toneSlope=1.34, toneToe=0.34,
      cg={ global={1.02,1.12,1.00,1.02}, shadows={1.00,1.12,1.00,1.00}, mids={1.02,1.12,1.00,1.02}, highs={1.00,1.08,1.00,1.00}, thr={0.04,0.92} } },

    -- 38
    { key="sp_preset_milkGlassOvercast",
      bMul=0.88, sMul=1.04, r=1.00, g=1.00, b=1.00,
      bloomMag=0.82, bloomThr=1.60, toneSlope=0.90, toneToe=0.46,
      cg={ global={0.92,0.98,1.02,0.98}, shadows={0.96,1.00,1.02,1.00}, mids={0.94,0.98,1.02,0.98}, highs={0.92,0.96,1.02,0.98}, thr={0.06,0.86} } },

    -- 39
    { key="sp_preset_azureSkyClean",
      bMul=1.28, sMul=1.06, r=0.98, g=1.04, b=1.18,
      bloomMag=0.48, bloomThr=2.10, toneSlope=1.08, toneToe=0.38,
      cg={ global={1.02,1.06,1.02,1.02}, shadows={1.00,1.06,1.02,1.00}, mids={1.02,1.06,1.02,1.02}, highs={1.00,1.04,1.02,1.00}, thr={0.05,0.88} } },

    -- 40
    { key="sp_preset_emberSunset",
      bMul=1.55, sMul=1.14, r=1.26, g=1.06, b=0.90,
      bloomMag=0.82, bloomThr=1.65, toneSlope=0.94, toneToe=0.36,
      cg={ global={1.12,1.10,1.00,1.06}, shadows={1.06,1.08,1.00,1.02}, mids={1.10,1.10,1.00,1.04}, highs={1.04,1.06,1.00,1.02}, thr={0.05,0.84} } },
}

------------------------------------------------------------------
-- ПРИМЕНЕНИЕ
------------------------------------------------------------------
local function applyPostFX(p)
    safeCall("setBloomMagnitude",        p.bloomMag or 0.5)
    safeCall("setBloomMaskThreshold",    p.bloomThr or 2.0)
    safeCall("setToneMappingCurveSlope", p.toneSlope or 1.0)
    safeCall("setToneMappingCurveToe",   p.toneToe or 0.4)

    if p.cg then
        local thr = p.cg.thr or {0.04, 0.85}
        safeCall("setColorGradingThresholds", thr[1], thr[2])

        local g = p.cg.global or {1,1,1,1}
        safeCall("setColorGradingGlobal", g[1], g[2], g[3], g[4])

        local s = p.cg.shadows or {1,1,1,1}
        safeCall("setColorGradingShadows", s[1], s[2], s[3], s[4])

        local m = p.cg.mids or {1,1,1,1}
        safeCall("setColorGradingMidtones", m[1], m[2], m[3], m[4])

        local h = p.cg.highs or {1,1,1,1}
        safeCall("setColorGradingHighlights", h[1], h[2], h[3], h[4])
    end
end

Lighting.updateAtmosphere = Utils.appendedFunction(Lighting.updateAtmosphere, function(self, dayMinutes)
    if not SunPresets.ready then return end
    local p = SunPresets.presets[SunPresets.selected]; if not p then return end

    if self.sunBrightnessScaleCurve ~= nil then
        local baseBrightness = self.sunBrightnessScaleCurve:get(dayMinutes)
        if baseBrightness ~= nil then
            safeCall("setSunBrightnessScale", baseBrightness * (p.bMul or 1.0))
        end
    end

    if self.sunSizeScaleCurve ~= nil then
        local baseSize = self.sunSizeScaleCurve:get(dayMinutes)
        if baseSize ~= nil then
            safeCall("setSunSizeScale", baseSize * (p.sMul or 1.0))
        end
    end

    if self.primaryExtraterrestrialColor ~= nil and self.primaryDynamicLightingScale ~= nil and self.sunLightId ~= nil then
        local r, g, b = self.primaryExtraterrestrialColor:get(dayMinutes)
        local dyn = self.primaryDynamicLightingScale:get(dayMinutes)
        if r and dyn then
            safeCall("setLightColor", self.sunLightId,
                (r or 1) * dyn * (p.r or 1),
                (g or 1) * dyn * (p.g or 1),
                (b or 1) * dyn * (p.b or 1)
            )
        end
    end

    applyPostFX(p)
end)

Lighting.updateCurves = Utils.appendedFunction(Lighting.updateCurves, function(self)
    SunPresets.ready = true
    SunPresets:applyNow()
end)

function SunPresets:applyNow()
    local env = g_currentMission and g_currentMission.environment
    local lighting = env and env.lighting
    local p = SunPresets.presets[SunPresets.selected]
    if not lighting or not p then return end

    applyPostFX(p)
    if SunPresets.ready then
        -- мягко пересчитаем освещение без скачка времени
        lighting:setDayTime(lighting.dayTime, true)
    end
end

------------------------------------------------------------------
-- СЕРИАЛИЗАЦИЯ (FogControl style + fallback)
------------------------------------------------------------------
function SunPresets:writeSettings()
    local newPath = _getSettingsPaths()
    createFolder(Utils.getFilename("modSettings/", getUserProfileAppPath()))

    local rootKey = "sunPresetsSettings"
    local xml = createXMLFile("sunPresets", newPath, rootKey)
    if xml == 0 or xml == nil then
        print("SunPresets: Failed to create XML for saving settings")
        return
    end

    local selected = clamp(SunPresets.selected or 1, 1, #SunPresets.presets)
    setXMLInt(xml, rootKey .. ".selected#value", selected)

    saveXMLFile(xml)
    delete(xml)
    print(string.format("SunPresets: Saved selected preset = %d to %s", selected, newPath))
end

function SunPresets:readSettings()
    local newPath, legacyPath = _getSettingsPaths()

    local function _tryLoad(path)
        if not fileExists(path) then return false end
        local xml = loadXMLFile("sunPresets", path)
        if xml == 0 or xml == nil then return false end

        local rootKey = "sunPresetsSettings"
        local propKey = rootKey .. ".selected#value"

        if hasXMLProperty(xml, propKey) then
            local val = getXMLInt(xml, propKey)
            if val ~= nil then
                SunPresets.selected = clamp(val, 1, #SunPresets.presets)
                delete(xml)
                print(string.format("SunPresets: Loaded selected preset = %d from %s", SunPresets.selected, path))
                return true
            end
        end

        delete(xml)
        return false
    end

    -- 1) новый путь
    if _tryLoad(newPath) then return end
    -- 2) старый путь (совместимость)
    if _tryLoad(legacyPath) then
        -- сразу пересохраним в новый путь
        SunPresets:writeSettings()
        return
    end

    -- 3) файла нет — пишем дефолт
    print("SunPresets: Settings file not found, writing defaults")
    SunPresets.selected = clamp(SunPresets.selected or 1, 1, #SunPresets.presets)
    SunPresets:writeSettings()
end

------------------------------------------------------------------
-- МЕНЮ (опционально; секция + слайдер)
------------------------------------------------------------------
local function buildPresetStrings()
    local strings = {}
    for i, p in ipairs(SunPresets.presets) do
        local name = g_i18n:getText(p.key)
        strings[i] = (name ~= "" and name) or ("Preset " .. tostring(i))
    end
    return strings
end

function SunPresets:addMenu()
    local inGameMenu  = g_gui.screenControllers[InGameMenu]
    if not inGameMenu then return end
    local settingsPage   = inGameMenu.pageSettings
    local layout = settingsPage and settingsPage.generalSettingsLayout
    if not layout then return end

    -- Заголовок
    local header
    for _, elem in ipairs(layout.elements) do
        if elem.name == "sectionHeader" then header = elem:clone(layout); break end
    end
    if header then
        header:setText(g_i18n:getText("sp_title") or "Lighting Presets")
    else
        local t = TextElement.new()
        t:applyProfile("fs25_settingsSectionHeader", true)
        t:setText(g_i18n:getText("sp_title") or "Lighting Presets")
        t.name = "sectionHeader"
        layout:addElement(t)
        header = t
    end
    header.focusId = FocusManager:serveAutoFocusId()
    table.insert(settingsPage.controlsList, header)
    SunPresets.CONTROLS["header"] = header

    -- Слайдер пресетов
    local original = settingsPage.multiVolumeVoiceBox
    local box = original:clone(layout)
    box.id = "SunPresetsBox"

    local opt = box.elements[1]
    opt.id = "SunPresetsSlider"
    opt.target = SunPresets
    opt:setCallback("onClickCallback", "onPresetChanged")
    opt:setDisabled(false)
    opt:setTexts(buildPresetStrings())
    opt:setState(SunPresets.selected)

    local tooltip = opt.elements[1]
    if tooltip then tooltip:setText(g_i18n:getText("sp_tooltip") or "Choose a global lighting preset") end

    local label = box.elements[2]
    label:setText(g_i18n:getText("sp_label") or "Lighting Preset")

    SunPresets.CONTROLS["slider"] = opt
    table.insert(settingsPage.controlsList, box)
    layout:invalidateLayout()
end

function SunPresets:onPresetChanged(state, _)
    SunPresets.selected = clamp(state, 1, #SunPresets.presets)
    SunPresets:writeSettings()
    SunPresets:applyNow()
    SunPresets:showToastForCurrent()
end

-- Инициализация UI при загрузке меню
local inGameMenu  = g_gui.screenControllers[InGameMenu]
local settingsPage   = inGameMenu and inGameMenu.pageSettings
local layout = settingsPage and settingsPage.generalSettingsLayout
if layout ~= nil then
    SunPresets:addMenu()
else
    Logging.info("SunPresets: settings layout will initialize when GUI loads.")
end

FocusManager.setGui = Utils.appendedFunction(FocusManager.setGui, function(_, gui)
    if gui == "ingameMenuSettings" then
        if not SunPresets.CONTROLS["slider"] then
            SunPresets:addMenu()
        else
            SunPresets.CONTROLS["slider"]:setTexts(buildPresetStrings())
            SunPresets.CONTROLS["slider"]:setState(SunPresets.selected)
        end
    end
end)

InGameMenuSettingsFrame.onFrameOpen = Utils.appendedFunction(InGameMenuSettingsFrame.onFrameOpen, function()
    if SunPresets.CONTROLS["slider"] then
        SunPresets.CONTROLS["slider"]:setTexts(buildPresetStrings())
        SunPresets.CONTROLS["slider"]:setState(SunPresets.selected)
    end
end)

------------------------------------------------------------------
-- HUD-тост + хоткеи Right Ctrl + ←/→
------------------------------------------------------------------
function SunPresets:showToastForCurrent()
    local p = SunPresets.presets[SunPresets.selected]; if not p then return end
    local name = g_i18n:getText(p.key); if name == "" then name = ("Preset " .. tostring(SunPresets.selected)) end
    SunPresets.toastText  = string.format(g_i18n:getText("sp_toast_format") or "Preset: %s", name)
    SunPresets.toastUntil = g_time + SunPresets.toastSec

    -- Если есть стандартный топ-нотификатор — используем его
    if g_currentMission and g_currentMission.hud and g_currentMission.hud.showTopNotification then
        g_currentMission.hud:showTopNotification(SunPresets.toastText)
    end
end

-- Рисуем свой тост, если топ-нотификатор недоступен
function SunPresets:draw()
    if SunPresets.toastUntil and g_time < SunPresets.toastUntil and (g_gui == nil or not g_gui:getIsGuiVisible()) then
        local text = SunPresets.toastText or ""
        if text ~= "" then
            -- простая шапка по центру
            setTextBold(true)
            setTextAlignment(RenderText.ALIGN_CENTER)
            setTextColor(1,1,1,1)
            renderText(0.5, 0.95, 0.022, text)
            setTextBold(false)
            setTextAlignment(RenderText.ALIGN_LEFT)
            setTextColor(1,1,1,1)
        end
    end
end

-- Хоткеи: перехватываем на уровне миссии
-- Важно: реагируем только, когда GUI закрыт, и ПРАВЫЙ Ctrl зажат.
function SunPresets:keyEvent(unicode, sym, modifier, isDown)
    if g_gui ~= nil and g_gui:getIsGuiVisible() then return end
    if not isDown then return end
    -- проверим, что зажат правый CTRL; имена констант зависят от билда — держим обе:
    local rightCtrlHeld = (Input.isKeyPressed and (Input.isKeyPressed(Input.KEY_rctrl) or Input.isKeyPressed(Input.KEY_RCTRL))) or false
    if not rightCtrlHeld then return end

    local goPrev = (sym == Input.KEY_left)  or (sym == Input.KEY_LEFT)
    local goNext = (sym == Input.KEY_right) or (sym == Input.KEY_RIGHT)
    if not (goPrev or goNext) then return end

    local newIndex = SunPresets.selected + (goNext and 1 or -1)
    if newIndex < 1 then newIndex = #SunPresets.presets end
    if newIndex > #SunPresets.presets then newIndex = 1 end

    SunPresets.selected = newIndex
    SunPresets:writeSettings()
    SunPresets:applyNow()
    SunPresets:showToastForCurrent()
end

------------------------------------------------------------------
-- ЖИЗНЕННЫЙ ЦИКЛ
------------------------------------------------------------------
function SunPresets:loadMap(name)
    SunPresets:readSettings()
    addConsoleCommand("gsSunPreset", "Set Sun Preset (index 1.."..tostring(#SunPresets.presets)..")",
        "consoleCommandSetPreset", SunPresets)
    if g_currentMission ~= nil then
        FSBaseMission.registerToLoadOnMapFinished(g_currentMission, SunPresets)
    end
end

function SunPresets:onLoadMapFinished()
    SunPresets:applyNow()
end

-- Консоль: gsSunPreset <index>
function SunPresets:consoleCommandSetPreset(idxStr)
    local idx = tonumber(idxStr or "")
    if not idx then
        return "Usage: gsSunPreset <index>"
    end
    SunPresets.selected = clamp(idx, 1, #SunPresets.presets)
    SunPresets:writeSettings()
    SunPresets:applyNow()
    SunPresets:showToastForCurrent()
    local key = SunPresets.presets[SunPresets.selected].key
    return string.format("Preset %d: %s", SunPresets.selected, g_i18n:getText(key) or key)
end
