-- LOC_makePaymentDialog.lua
-- Line of Credit mod - Make Payment dialog

LOC_makePaymentDialog = {}
local LOC_makePaymentDialog_mt = Class(LOC_makePaymentDialog, MessageDialog)

function LOC_makePaymentDialog.new(target, customMt)
    local self = MessageDialog.new(target, customMt or LOC_makePaymentDialog_mt)
    self.creditLineManager = nil -- Will be set when showing dialog
    self.callback = nil
    self.creditLine = nil
    return self
end

function LOC_makePaymentDialog:onCreate()
    -- Try to get elements using getDescendantById if direct access fails
    self.paymentAmountInput = self:getDescendantById("paymentAmountInput") or self.elements.paymentAmountInput
    self.currentBalanceField = self:getDescendantById("currentBalanceField") or self.elements.currentBalanceField
    self.newBalanceField = self:getDescendantById("newBalanceField") or self.elements.newBalanceField
    self.minimumPaymentField = self:getDescendantById("minimumPaymentField") or self.elements.minimumPaymentField
    self.payFullButton = self:getDescendantById("payFullButton") or self.elements.payFullButton
    
end

function LOC_makePaymentDialog:onOpen()
    MessageDialog.onOpen(self)
    
    if self.paymentAmountInput then
        self.paymentAmountInput:setText("")
        FocusManager:setFocus(self.paymentAmountInput)
    end
    
    self:updateInfoDisplay()
end

function LOC_makePaymentDialog:setCreditLine(creditLine, creditLineManager)
    self.creditLine = creditLine
    self.creditLineManager = creditLineManager
end

function LOC_makePaymentDialog:setCallback(callback)
    self.callback = callback
end

function LOC_makePaymentDialog:onTextChanged(element, text)
    if element == self.paymentAmountInput then
        self:updateInfoDisplay()
    end
end

function LOC_makePaymentDialog:updateInfoDisplay()
    if not self.creditLine then
        return
    end
    
    local paymentAmountText = self.paymentAmountInput and self.paymentAmountInput:getText() or ""
    local paymentAmount = tonumber(paymentAmountText) or 0
    
    -- Update current balance
    local currentBalance = self.creditLine.currentBalance or 0
    if self.currentBalanceField then
        self.currentBalanceField:setText(string.format("Current Balance: $%s", 
            g_i18n:formatMoney(currentBalance)))
    end
    
    -- Calculate and display remaining monthly payment
    local remainingMonthly = 0
    if self.creditLine.getRemainingMonthlyPayment then
        remainingMonthly = self.creditLine:getRemainingMonthlyPayment()
    else
        -- Fallback for older credit lines without the new method
        remainingMonthly = self.creditLine.monthlyPayment or math.max(25, currentBalance * 0.02)
        if currentBalance < 25 then
            remainingMonthly = currentBalance
        end
    end
    
    if self.minimumPaymentField then
        if remainingMonthly > 0 then
            self.minimumPaymentField:setText(string.format("Remaining Monthly Payment: $%s", 
                g_i18n:formatMoney(remainingMonthly)))
            self.minimumPaymentField:setTextColor(1, 0.8, 0.2, 1) -- Yellow to indicate payment due
        elseif currentBalance > 0 then
            self.minimumPaymentField:setText("Monthly payment satisfied")
            self.minimumPaymentField:setTextColor(0.2, 1, 0.2, 1) -- Green when satisfied
        else
            self.minimumPaymentField:setText("No payment required")
            self.minimumPaymentField:setTextColor(1, 1, 1, 1) -- White
        end
    end
    
    -- Update new balance after payment
    local newBalance = math.max(0, currentBalance - paymentAmount)
    if self.newBalanceField then
        if paymentAmount > 0 then
            self.newBalanceField:setText(string.format("New Balance: $%s", 
                g_i18n:formatMoney(newBalance)))
            
            -- Color code based on payment amount vs remaining monthly
            if paymentAmount < remainingMonthly and currentBalance > 0 and remainingMonthly > 0 then
                self.newBalanceField:setTextColor(1, 0.8, 0.2, 1) -- Yellow for below remaining monthly
            else
                self.newBalanceField:setTextColor(1, 1, 1, 1) -- White otherwise
            end
        else
            self.newBalanceField:setText("New Balance: --")
            self.newBalanceField:setTextColor(1, 1, 1, 1)
        end
    end
    
    -- Update pay full button text with payoff amount
    if self.payFullButton then
        local payoffAmount = self.creditLine:getPayoffAmount() or currentBalance
        self.payFullButton:setText(string.format("Pay Full Balance ($%s)", 
            g_i18n:formatMoney(payoffAmount)))
    end
end

function LOC_makePaymentDialog:onEnterPressed(element)
    if element == self.paymentAmountInput then
        self:onClickOk()
    end
end

function LOC_makePaymentDialog:onClickPayFull()
    if self.paymentAmountInput and self.creditLine then
        -- Use payoff amount which includes interest
        local payoffAmount = self.creditLine:getPayoffAmount() or 0
        -- Use exact amount, not ceiling to avoid overpayment
        self.paymentAmountInput:setText(tostring(math.floor(payoffAmount + 0.5)))  -- Round to nearest dollar
        self:updateInfoDisplay()
    end
end

function LOC_makePaymentDialog:onClickOk()
    
    -- Try different methods to get the text
    local paymentAmountText = ""
    if self.paymentAmountInput then
        if self.paymentAmountInput.getText then
            paymentAmountText = self.paymentAmountInput:getText()
        elseif self.paymentAmountInput.text then
            paymentAmountText = self.paymentAmountInput.text
        end
    end
    
    local paymentAmount = tonumber(paymentAmountText)
    
    if not paymentAmount or paymentAmount <= 0 then
        if g_currentMission.hud then
            g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL, 
                "Please enter a valid payment amount.")
        end
        return
    end
    
    if paymentAmount > g_currentMission:getMoney() then
        if g_currentMission.hud then
            g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL, 
                string.format("Insufficient funds. You have $%s available.", 
                    g_i18n:formatMoney(g_currentMission:getMoney())))
        end
        return
    end
    
    -- Check if this is a payoff amount (includes interest)
    local currentBalance = self.creditLine.currentBalance or 0
    local payoffAmount = self.creditLine:getPayoffAmount() or currentBalance
    
    -- Allow payment up to the payoff amount (which includes interest)
    if paymentAmount > payoffAmount then
        paymentAmount = payoffAmount
        if g_currentMission.hud then
            g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_INFO, 
                string.format("Payment adjusted to full payoff amount of $%s", 
                    g_i18n:formatMoney(payoffAmount)))
        end
    end
    
    self:close()
    
    if self.callback then
        self.callback(paymentAmount)
    end
end

function LOC_makePaymentDialog:onClickCancel()
    self:close()
end