-- Line of Credit In-Game Menu
-- Author: sockswayup
-- Version: 1.0.4.0
-- Based on Enhanced Loan System pattern

LOC_inGameMenuCredit = {}

local LOC_inGameMenuCredit_mt = Class(LOC_inGameMenuCredit, TabbedMenuFrameElement)

function LOC_inGameMenuCredit.new(i18n, messageCenter, customMt)
    local self = LOC_inGameMenuCredit:superClass().new(nil, customMt or LOC_inGameMenuCredit_mt)
    
    self.hasCustomMenuButtons = true
    self.messageCenter = messageCenter
    self.i18n = i18n
    
    -- Initialize arrays
    self.currentCreditLines = {}
    
    -- UI elements will be set when the GUI is loaded
    self.creditLimitText = nil
    self.availableCreditText = nil
    self.currentBalanceText = nil
    self.interestRateText = nil
    self.monthlyPaymentText = nil
    
    return self
end

function LOC_inGameMenuCredit:initialize()
    self.backButtonInfo = {
        inputAction = InputAction.MENU_BACK
    }
    
    -- Initial empty menu buttons - will be populated in updateButtons()
    self.menuButtons = {self.backButtonInfo}
    self:setMenuButtonInfo(self.menuButtons)
end

function LOC_inGameMenuCredit:onGuiSetupFinished()
    LOC_inGameMenuCredit:superClass().onGuiSetupFinished(self)
    local creditTable = self:getDescendantById("creditTable")
    if creditTable then
        creditTable:setDataSource(self)
        creditTable:setDelegate(self)
    end
end

function LOC_inGameMenuCredit:onFrameOpen()
    LOC_inGameMenuCredit:superClass().onFrameOpen(self)
    
    -- Get available credit value element for header
    self.availableCredit = self:getDescendantById("availableCredit")
    
    -- Get the credit table
    self.creditTable = self:getDescendantById("creditTable")
    
    self:updateContent()
    self:updateButtons()
    
    -- Don't set focus to credit table - let menu navigation work properly
end

function LOC_inGameMenuCredit:onFrameActivated()
    -- Called when the frame becomes active (e.g., switching tabs)
    self:updateContent()
    self:updateButtons()
end

function LOC_inGameMenuCredit:updateContent()
    if not g_currentMission then
        return
    end
    
    local farmId = g_currentMission:getFarmId() or 1
    
    -- Debug: Check if g_loc_manager exists and has the method
    if not g_loc_manager then
        self.currentCreditLines = {}
    elseif not g_loc_manager.currentCreditLines then
        self.currentCreditLines = {}
    else
        self.currentCreditLines = g_loc_manager:currentCreditLines(farmId) or {}
    end
    
    -- Update available credit in header
    if self.availableCredit then
        if #self.currentCreditLines > 0 then
            local creditLine = self.currentCreditLines[1]
            local available = creditLine.getAvailableCredit and creditLine:getAvailableCredit() or (creditLine.creditLimit - creditLine.currentBalance)
            self.availableCredit:setText(self.i18n:getText("LOC_AVAILABLE_CREDIT") .. ": " .. g_i18n:formatMoney(available, 0, true, true))
        else
            self.availableCredit:setText(self.i18n:getText("LOC_AVAILABLE_CREDIT") .. ": $0")
        end
    end
    
    -- Reload the table
    if self.creditTable then
        self.creditTable:reloadData()
    end
end

function LOC_inGameMenuCredit:updateButtons()
    if not g_currentMission then
        return
    end
    
    local farmId = g_currentMission:getFarmId() or 1
    
    -- Debug logging
    
    if self.currentCreditLines and #self.currentCreditLines > 0 then
    end
    
    -- Create fresh button configurations based on state
    local buttons = {}
    table.insert(buttons, self.backButtonInfo)
    
    if not self.currentCreditLines or #self.currentCreditLines == 0 then
        -- No credit line - show apply and settings buttons
        
        -- Create a fresh apply button
        local applyBtn = {
            inputAction = InputAction.MENU_ACTIVATE,
            text = self.i18n:getText("LOC_APPLY_FOR_CREDIT") or "Apply for Credit Line",
            callback = function() self:onApplyForCreditClicked() end,
            disabled = false
        }
        table.insert(buttons, applyBtn)
    else
        -- Has credit line - show draw funds button
        
        -- Create a fresh draw funds button
        local drawBtn = {
            inputAction = InputAction.MENU_ACTIVATE,
            text = self.i18n:getText("LOC_DRAW_FUNDS") or "Draw Funds",
            callback = function() self:onDrawFundsClicked() end,
            disabled = false
        }
        table.insert(buttons, drawBtn)
        
        -- Add make payment button if there's a balance
        local currentBalance = self.currentCreditLines[1].currentBalance or 0
        
        if currentBalance > 0.01 then  -- Use small threshold to handle floating point
            local paymentBtn = {
                inputAction = InputAction.MENU_EXTRA_1,
                text = self.i18n:getText("LOC_MAKE_PAYMENT") or "Make Payment",
                callback = function() self:onMakePaymentClicked() end,
                disabled = false
            }
            table.insert(buttons, paymentBtn)
        else
            -- Add close credit line button when balance is zero or very small
            local closeBtn = {
                inputAction = InputAction.MENU_EXTRA_1,
                text = self.i18n:getText("LOC_CLOSE_CREDIT") or "Close Credit Line",
                callback = function() self:onCloseCreditLineClicked() end,
                disabled = false
            }
            table.insert(buttons, closeBtn)
        end
    end
    
    -- Always add settings button with current mode
    -- Ensure manager is initialized
    if not g_loc_manager then
        print("[Line of Credit] ERROR: g_loc_manager is nil! Reinitializing...")
        g_loc_manager = LOC_manager.new()
    end
    local currentMode = g_loc_manager.properties.calculationMode or "default"
    local modeText = currentMode == "sandbox" and "Sandbox" or "Default"
    local settingsBtn = {
        inputAction = InputAction.MENU_EXTRA_2,
        text = string.format("Mode: %s (Toggle)", modeText),
        callback = function() self:onSettingsClicked() end,
        disabled = false
    }
    table.insert(buttons, settingsBtn)
    
    -- Update the menu buttons
    self.menuButtons = buttons
    self:setMenuButtonInfo(self.menuButtons)
    self:setMenuButtonInfoDirty()
    
end

function LOC_inGameMenuCredit:onPayoffBalanceClicked()
    if not g_currentMission then
        return
    end
    local farmId = g_currentMission:getFarmId() or 1
    local creditLines = g_loc_manager and g_loc_manager.currentCreditLines and g_loc_manager:currentCreditLines(farmId) or {}
    
    if #creditLines > 0 and creditLines[1].currentBalance > 0 then
        local balance = creditLines[1].currentBalance
        self:showMakePaymentDialog(creditLines[1])
    end
end

-- This method is replaced by updateContent and list DataSource methods

function LOC_inGameMenuCredit:onApplyForCreditClicked()
    self:showCreateCreditLineDialog()
end

function LOC_inGameMenuCredit:onDrawFundsClicked()
    if not g_currentMission then
        return
    end
    local farmId = g_currentMission:getFarmId() or 1
    local creditLines = g_loc_manager:currentCreditLines(farmId)
    
    if #creditLines > 0 then
        -- Show dialog to draw from existing credit line
        self:showDrawFundsDialog(creditLines[1])
    else
    end
end

function LOC_inGameMenuCredit:onMakePaymentClicked()
    if not g_currentMission then
        print("[Line of Credit] No mission available")
        return
    end
    local farmId = g_currentMission:getFarmId() or 1
    local creditLines = g_loc_manager:currentCreditLines(farmId)
    
    if #creditLines > 0 and creditLines[1].currentBalance > 0 then
        self:showMakePaymentDialog(creditLines[1])
    end
end

function LOC_inGameMenuCredit:onSettingsClicked()
    self:showCreditModeDialog()
end

function LOC_inGameMenuCredit:onCloseCreditLineClicked()
    if not g_currentMission then
        return
    end
    local farmId = g_currentMission:getFarmId() or 1
    local creditLines = g_loc_manager:currentCreditLines(farmId)
    
    if #creditLines > 0 and creditLines[1].currentBalance <= 0.01 then
        -- For now, just close directly with a notification
        -- TODO: Create a proper confirmation dialog XML like the other dialogs
        local creditLine = creditLines[1]
        local creditLimit = creditLine.creditLimit
        local monthsUsed = creditLine.paymentsMade
        local totalMonths = creditLine.paymentTerm
        
        -- Show warning notification about closing
        g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_INFO,
            string.format("Closing credit line ($%s limit, used %d of %d months)",
                g_i18n:formatMoney(creditLimit), monthsUsed, totalMonths))
        
        -- Close the credit line
        local success, message = g_loc_manager:closeCreditLine(creditLine)
        if success then
            g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_OK,
                "Credit line closed successfully. You can apply for a new one.")
            self:updateContent()
            self:updateButtons()
        else
            g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL,
                message or "Failed to close credit line")
        end
    else
        g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL,
            "Cannot close credit line - outstanding balance must be zero")
    end
end

function LOC_inGameMenuCredit:showCreditModeDialog()
    -- Simply toggle between Default and Sandbox mode
    local currentMode = g_loc_manager.properties.calculationMode or "default"
    local newMode = "default"
    local modeDescription = ""
    
    if currentMode == "default" or currentMode == "realistic" or currentMode == "relaxed" or currentMode == "fixed" then
        -- Switch to sandbox
        newMode = "sandbox"
        modeDescription = "Sandbox Mode - Unlimited credit available"
    else
        -- Switch to default
        newMode = "default"
        modeDescription = "Default Mode - Credit based on collateral (vehicles, land, money)"
    end
    
    -- Apply the new mode
    self:changeCreditMode(newMode)
    
    -- Show notification of the change
    g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_INFO,
        string.format("Credit mode changed to: %s", modeDescription))
end

function LOC_inGameMenuCredit:changeCreditMode(newMode)
    -- Update the property
    g_loc_manager.properties.calculationMode = newMode
    
    -- Apply mode-specific settings
    if newMode == "sandbox" then
        -- Sandbox mode - no limits
    else
        -- Default mode - use standard collateral-based calculation
        g_loc_manager.properties.maxCreditToCollateralRatio = 80
        g_loc_manager.properties.vehicleCollateralPercentage = 70
        g_loc_manager.properties.farmlandCollateralPercentage = 60
    end
    
    -- Save to config file
    self:saveCreditModeToConfig(newMode)
    
    -- Update display and buttons to reflect new limits and mode
    self:updateContent()
    self:updateButtons()  -- Force button refresh to show new mode
end

function LOC_inGameMenuCredit:saveCreditModeToConfig(mode)
    if not g_currentModDirectory then
        return
    end
    
    local configFile = g_currentModDirectory .. "config/creditConfig.xml"
    local xmlFile = XMLFile.create("creditConfig", configFile)
    
    if xmlFile then
        xmlFile:setString("configuration.mode", mode)
        
        -- Keep other settings
        xmlFile:setFloat("configuration.creditLimit", g_loc_manager.properties.fixedCreditLimit or 5000000)
        xmlFile:setFloat("configuration.maxCreditToCollateralRatio", g_loc_manager.properties.maxCreditToCollateralRatio)
        xmlFile:setFloat("configuration.vehicleCollateralPercentage", g_loc_manager.properties.vehicleCollateralPercentage)
        xmlFile:setFloat("configuration.farmlandCollateralPercentage", g_loc_manager.properties.farmlandCollateralPercentage)
        xmlFile:setFloat("configuration.baseInterestRate", g_loc_manager.properties.baseInterestRate)
        xmlFile:setFloat("configuration.minInterestRate", g_loc_manager.properties.minInterestRate)
        xmlFile:setFloat("configuration.maxInterestRate", g_loc_manager.properties.maxInterestRate)
        xmlFile:setFloat("configuration.rateFluctuationRange", g_loc_manager.properties.rateFluctuationRange)
        xmlFile:setInt("configuration.paymentTerm", g_loc_manager.properties.defaultPaymentTermYears * 12)
        xmlFile:setInt("configuration.paymentGracePeriod", g_loc_manager.properties.gracePeriodDays)
        xmlFile:setFloat("configuration.lateFeeAmount", g_loc_manager.properties.lateFeeAmount)
        xmlFile:setBool("configuration.enableNotifications", g_loc_manager.properties.enableNotifications)
        xmlFile:setBool("configuration.showHudOverlay", g_loc_manager.properties.showHudOverlay)
        
        xmlFile:save()
        xmlFile:delete()
        
    end
end

function LOC_inGameMenuCredit:showCreateCreditLineDialog()
    if not g_currentMission then
        return
    end
    
    local farmId = g_currentMission:getFarmId()
    if not farmId or farmId == 0 then
        g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL, 
            "You must be a member of a farm to apply for a line of credit")
        return
    end
    
    local maxCredit = g_loc_manager:maxCreditLimitForFarm(farmId)
    
    if maxCredit > 0 then
        -- Get the dialog from g_gui
        local dialog = g_gui.guis["LOC_applyCreditDialog"]
        if dialog and dialog.target then
            dialog.target:setAvailableProperties(maxCredit, g_loc_manager.properties.baseInterestRate, g_loc_manager.properties.defaultPaymentTermYears)
            dialog.target:setCallback(function(target, success, creditAmount)
                if success and creditAmount then
                    -- Create credit line directly like ELS does
                    print("[Line of Credit] Creating credit line directly (ELS pattern)")
                    local creditLine = LOC_creditLine.new(g_currentMission:getIsServer(), g_currentMission:getIsClient())
                    creditLine:init(farmId, creditAmount, g_loc_manager.properties.baseInterestRate, 
                        g_loc_manager.properties.defaultPaymentTermYears, g_loc_manager.properties.dynamicInterestRate)
                    
                    g_loc_manager:addCreditLine(creditLine)
                    
                    -- Show notification
                    if g_currentMission.hud then
                        g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_OK, 
                            string.format("Line of Credit Approved: %s", g_i18n:formatMoney(creditAmount)))
                    end
                    
                    
                    -- Force update display
                    self:updateContent()
                    self:updateButtons()
                end
            end, self)
            
            g_gui:showDialog("LOC_applyCreditDialog")
        else
        end
    else
        -- Check if it's because we're in default mode with no collateral
        local currentMode = g_loc_manager.properties.calculationMode or "default"
        if currentMode == "default" then
            g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL, 
                "Insufficient collateral. You need vehicles, land, or money. Try Sandbox mode for unlimited credit.")
        else
            g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL, 
                "Unable to apply for credit line")
        end
    end
end

function LOC_inGameMenuCredit:showDrawFundsDialog(creditLine)
    local available = creditLine:getAvailableCredit()
    
    if available <= 0 then
        g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL, 
            "No credit available to draw")
        return
    end
    
    -- Use the new draw funds dialog
    if g_locDrawFundsDialog then
        g_locDrawFundsDialog:setCreditLine(creditLine, g_loc_manager)
        g_locDrawFundsDialog:setCallback(function(drawAmount)
            local success, message = g_loc_manager:drawFunds(creditLine, drawAmount)
            
            if success then
                self:updateContent()
                self:updateButtons()
                if g_currentMission.hud then
                    g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_OK, 
                        string.format("Drew %s from credit line", g_i18n:formatMoney(drawAmount)))
                end
            else
                if g_currentMission.hud then
                    g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL, 
                        "Draw failed: " .. message)
                end
            end
        end)
        
        g_gui:showDialog("LOC_drawFundsDialog")
    else
    end
end

function LOC_inGameMenuCredit:showMakePaymentDialog(creditLine)
    local balance = creditLine.currentBalance
    local monthlyPayment = creditLine.monthlyPayment
    
    if balance <= 0 then
        g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_INFO, 
            "No balance to pay")
        return
    end
    
    -- Use the new make payment dialog
    if g_locMakePaymentDialog then
        g_locMakePaymentDialog:setCreditLine(creditLine, g_loc_manager)
        g_locMakePaymentDialog:setCallback(function(paymentAmount)
            local success, message = g_loc_manager:makePayment(creditLine, paymentAmount)
            
            if success then
                self:updateContent()
                
                -- Debug: Check the balance after payment
                if creditLine then
                end
                
                -- Force button update
                self:updateButtons()
                
                if g_currentMission.hud then
                    g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_OK, 
                        string.format("Payment of %s applied", g_i18n:formatMoney(paymentAmount)))
                end
            else
                if g_currentMission.hud then
                    g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL, 
                        "Payment failed: " .. message)
                end
            end
        end)
        
        g_gui:showDialog("LOC_makePaymentDialog")
    else
    end
end

-- DataSource methods for list
function LOC_inGameMenuCredit:getNumberOfSections()
    return 1  -- Just one section for credit lines
end

function LOC_inGameMenuCredit:getNumberOfItemsInSection(list, section)
    if section == 1 then
        -- Show at least one row even if no credit lines
        return math.max(1, #self.currentCreditLines)
    end
    return 0
end

function LOC_inGameMenuCredit:getTitleForSectionHeader(list, section)
    return ""  -- No section headers needed
end

function LOC_inGameMenuCredit:populateCellForItemInSection(list, section, index, cell)
    if section ~= 1 then
        return
    end
    
    local creditLine = self.currentCreditLines[index]
    
    if creditLine then
        
        -- Populate with actual credit line data
        cell:getAttribute("creditLimit"):setText(g_i18n:formatMoney(creditLine.creditLimit, 0, true, true))
        cell:getAttribute("interestRate"):setText(string.format("%.2f%%", (creditLine.currentInterestRate or 0.06) * 100))
        
        -- Show monthly payment or indicate no balance
        if creditLine.currentBalance > 0 then
            cell:getAttribute("monthlyPayment"):setText(g_i18n:formatMoney(creditLine.monthlyPayment or 0, 0, true, true))
        else
            cell:getAttribute("monthlyPayment"):setText("$0 (No Balance)")
        end
        
        -- Show current balance (amount borrowed)
        cell:getAttribute("nextPayment"):setText(g_i18n:formatMoney(creditLine.currentBalance or 0, 0, true, true))
        
        -- Show months remaining instead of total interest
        local monthsRemaining = creditLine:getRemainingTerm()
        if monthsRemaining > 0 then
            cell:getAttribute("totalInterest"):setText(string.format("%d months", monthsRemaining))
        elseif creditLine.currentBalance > 0 then
            cell:getAttribute("totalInterest"):setText("DUE NOW")
        else
            cell:getAttribute("totalInterest"):setText("--")
        end
    else
        -- Show placeholder values when no credit line
        cell:getAttribute("creditLimit"):setText("$0")
        cell:getAttribute("interestRate"):setText("6.00%")
        cell:getAttribute("monthlyPayment"):setText("$0")
        cell:getAttribute("nextPayment"):setText("$0")
        cell:getAttribute("totalInterest"):setText("--")
    end
end

function LOC_inGameMenuCredit:onListSelectionChanged(list, section, index)
    -- Could be used to select different credit lines in future
    self.selectedCreditLine = self.currentCreditLines[index]
end
