-- Line of Credit Application Dialog
-- Author: sockswayup
-- Based on Enhanced Loan System pattern

LOC_applyCreditDialog = {}
local LOC_applyCreditDialog_mt = Class(LOC_applyCreditDialog, MessageDialog)

function LOC_applyCreditDialog.new(target, custom_mt, i18n)
    local self = MessageDialog.new(target, custom_mt or LOC_applyCreditDialog_mt)
    
    self.i18n = i18n
    self.maxCreditAmount = 0
    self.interestRate = 6.0
    self.paymentTermYears = 10
    
    return self
end

function LOC_applyCreditDialog:onOpen()
    LOC_applyCreditDialog:superClass().onOpen(self)
    
    -- Get UI elements
    self.creditAmountInput = self:getDescendantById("creditAmountInput")
    self.creditAmountInputText = self:getDescendantById("creditAmountInputText")
    self.interestRateField = self:getDescendantById("interestRateField")
    self.paymentTermField = self:getDescendantById("paymentTermField")
    self.monthlyPaymentField = self:getDescendantById("monthlyPaymentField")
    self.yesButton = self:getDescendantById("yesButton")
    
    self:resetUI()
    
    if self.creditAmountInputText then
        self.creditAmountInputText:setText(string.format("%s (Max. %s):", 
            "Credit Amount", 
            g_i18n:formatMoney(self.maxCreditAmount)))
    end
    
    if self.creditAmountInput then
        FocusManager:setFocus(self.creditAmountInput)
    end
end

function LOC_applyCreditDialog:resetUI()
    if self.creditAmountInput then
        self.creditAmountInput:setText("")
        self.creditAmountInput.lastValidText = ""
    end
    
    if self.yesButton then
        self.yesButton:setDisabled(true)
    end
    
    if self.interestRateField then
        self.interestRateField:setText(string.format("Interest Rate: %.1f%%", self.interestRate))
    end
    
    if self.paymentTermField then
        self.paymentTermField:setText(string.format("Payment Term: %d years", self.paymentTermYears))
    end
    
    if self.monthlyPaymentField then
        self.monthlyPaymentField:setText("Est. Monthly Payment: -")
    end
end

function LOC_applyCreditDialog:setAvailableProperties(maxCreditAmount, interestRate, paymentTermYears)
    self.maxCreditAmount = math.max(maxCreditAmount, 0)
    self.interestRate = interestRate
    self.paymentTermYears = paymentTermYears
end

-- Alias for compatibility
function LOC_applyCreditDialog:setMaxCredit(maxCredit)
    self.maxCreditAmount = maxCredit
end

function LOC_applyCreditDialog:setCreditLineManager(manager)
    self.creditLineManager = manager
end

function LOC_applyCreditDialog:setCallback(callbackFunc, target)
    self.callbackFunc = callbackFunc
    self.target = target
end

function LOC_applyCreditDialog:onClickOk()
    self:sendCallback(true)
end

function LOC_applyCreditDialog:onClickCancel()
    self:sendCallback(false)
end

function LOC_applyCreditDialog:sendCallback(success)
    self:close()
    
    if self.callbackFunc ~= nil then
        if self.target ~= nil then
            local amountInput = tonumber(self.creditAmountInput.lastValidText)
            self.callbackFunc(self.target, success, amountInput)
        end
    end
end

function LOC_applyCreditDialog:onEnterPressed(element)
    if element == self.creditAmountInput then
        self:onClickOk()
    end
end

function LOC_applyCreditDialog:onTextChanged(element, text)
    if text ~= "" then
        if text ~= element.lastValidText and tonumber(text) ~= nil then
            local value = text
            if element.id == "creditAmountInput" then
                local currentValue = tonumber(value)
                if currentValue > self.maxCreditAmount then
                    value = self.maxCreditAmount
                end
            end
            
            local formattedValue = string.format("%.0f", value)
            element.lastValidText = formattedValue
            element:setText(formattedValue)
        else
            element:setText(element.lastValidText)
        end
    else
        element.lastValidText = ""
    end
    
    self:updateInfoIfNeeded()
    self:disableAcceptButtonIfNeeded()
end

function LOC_applyCreditDialog:disableAcceptButtonIfNeeded()
    if self.creditAmountInput.lastValidText ~= nil and
        self.creditAmountInput.lastValidText ~= "" and
        tonumber(self.creditAmountInput.lastValidText) > 0
    then
        self.yesButton:setDisabled(false)
    else
        self.yesButton:setDisabled(true)
    end
end

function LOC_applyCreditDialog:updateInfoIfNeeded()
    if self.creditAmountInput.lastValidText ~= nil and
        self.creditAmountInput.lastValidText ~= "" and
        tonumber(self.creditAmountInput.lastValidText) > 0
    then
        local amount = tonumber(self.creditAmountInput.lastValidText)
        
        -- Calculate estimated monthly payment if balance is fully drawn
        local monthlyRate = self.interestRate / 100 / 12
        local months = self.paymentTermYears * 12
        
        if monthlyRate > 0 and months > 0 then
            local factor = math.pow(1 + monthlyRate, months)
            local monthlyPayment = amount * (monthlyRate * factor) / (factor - 1)
            
            self.monthlyPaymentField:setText(string.format("%s: %s", 
                self.i18n:getText("LOC_MONTHLY_PAYMENT") or "Est. Monthly Payment", 
                g_i18n:formatMoney(monthlyPayment)))
        end
    else
        self.monthlyPaymentField:setText(string.format("%s: -", 
            self.i18n:getText("LOC_MONTHLY_PAYMENT") or "Est. Monthly Payment"))
    end
end