-- Line of Credit Properties
-- Author: sockswayup  
-- Version: 1.0.4.0
-- Simplified version without network sync

LOC_properties = {}

local LOC_properties_mt = Class(LOC_properties)

function LOC_properties.new(isServer, isClient, customMt)
    local self = setmetatable({}, customMt or LOC_properties_mt)
    
    self.isServer = isServer
    self.isClient = isClient
    
    -- Default credit line properties
    self.calculationMode = "realistic" -- Default mode
    self.baseInterestRate = 6.0 -- 6%
    self.rateFluctuationRange = 2.0 -- +/- 2%
    self.minInterestRate = 1.0 -- 1% minimum
    self.maxInterestRate = 15.0 -- 15% maximum
    self.dynamicInterestRate = true
    
    self.defaultCreditLimit = 1000000 -- $1M default
    self.fixedCreditLimit = 5000000 -- $5M for fixed mode
    self.maxCreditToCollateralRatio = 80.0 -- 80% of collateral
    self.vehicleCollateralPercentage = 70.0 -- 70% of vehicle value
    self.farmlandCollateralPercentage = 80.0 -- 80% of farmland value
    
    self.defaultPaymentTermYears = 10 -- 10 years
    self.minPaymentTermYears = 1 -- 1 year minimum
    self.maxPaymentTermYears = 30 -- 30 years maximum
    
    self.lateFeeAmount = 50.0 -- $50 late fee
    self.gracePeriodDays = 7 -- 7 day grace period
    
    self.enableNotifications = true
    self.showHudOverlay = true
    
    return self
end

function LOC_properties:register()
    print("[Line of Credit] Properties registered successfully")
end

function LOC_properties:delete()
    -- Cleanup if needed
end

function LOC_properties:loadFromConfig()
    local modDir = g_currentModDirectory or ""
    if modDir == "" then
        print("[Line of Credit] Warning: g_currentModDirectory is not set")
        return
    end
    local configFile = modDir .. "config/creditConfig.xml"
    
    if fileExists(configFile) then
        local xmlFile = XMLFile.load("creditConfig", configFile)
        
        if xmlFile then
            -- Load calculation mode (default to "realistic")
            self.calculationMode = xmlFile:getString("configuration.mode", "realistic")
            
            -- Load collateral settings for realistic mode
            if self.calculationMode == "realistic" or self.calculationMode == "relaxed" then
                self.maxCreditToCollateralRatio = xmlFile:getFloat("configuration.maxCreditToCollateralRatio", 80.0)
                self.vehicleCollateralPercentage = xmlFile:getFloat("configuration.vehicleCollateralPercentage", 70.0)
                self.farmlandCollateralPercentage = xmlFile:getFloat("configuration.farmlandCollateralPercentage", 80.0)
                
                -- For relaxed mode, boost the percentages
                if self.calculationMode == "relaxed" then
                    self.maxCreditToCollateralRatio = self.maxCreditToCollateralRatio * 2 -- 160% instead of 80%
                    self.vehicleCollateralPercentage = math.min(100, self.vehicleCollateralPercentage * 1.5)
                    self.farmlandCollateralPercentage = math.min(100, self.farmlandCollateralPercentage * 1.5)
                end
            end
            
            -- Load fixed limits for fixed/unlimited modes
            self.fixedCreditLimit = xmlFile:getFloat("configuration.creditLimit", 5000000)
            
            -- Load interest settings
            self.baseInterestRate = xmlFile:getFloat("configuration.baseInterestRate", 6.0)
            self.minInterestRate = xmlFile:getFloat("configuration.minInterestRate", 1.0)
            self.maxInterestRate = xmlFile:getFloat("configuration.maxInterestRate", 15.0)
            self.dynamicInterestRate = xmlFile:getBool("configuration.dynamicInterestRate", true)
            self.rateFluctuationRange = xmlFile:getFloat("configuration.rateFluctuationRange", 2.0)
            
            -- Load payment settings
            local paymentTermMonths = xmlFile:getInt("configuration.paymentTerm", 120)
            self.defaultPaymentTermYears = math.floor(paymentTermMonths / 12)
            self.gracePeriodDays = xmlFile:getInt("configuration.paymentGracePeriod", 7)
            self.lateFeeAmount = xmlFile:getFloat("configuration.lateFeeAmount", 50)
            
            -- Load UI settings
            self.enableNotifications = xmlFile:getBool("configuration.enableNotifications", true)
            self.showHudOverlay = xmlFile:getBool("configuration.showHudOverlay", true)
            
            xmlFile:delete()
            
        end
    else
        print("[Line of Credit] No config file found, using defaults")
    end
end

function LOC_properties:saveToXML(xmlFile, key)
    xmlFile:setString(key .. "#calculationMode", self.calculationMode)
    xmlFile:setFloat(key .. "#baseInterestRate", self.baseInterestRate)
    xmlFile:setFloat(key .. "#rateFluctuationRange", self.rateFluctuationRange)
    xmlFile:setFloat(key .. "#minInterestRate", self.minInterestRate)
    xmlFile:setFloat(key .. "#maxInterestRate", self.maxInterestRate)
    xmlFile:setBool(key .. "#dynamicInterestRate", self.dynamicInterestRate)
    xmlFile:setFloat(key .. "#fixedCreditLimit", self.fixedCreditLimit)
    xmlFile:setFloat(key .. "#maxCreditToCollateralRatio", self.maxCreditToCollateralRatio)
    xmlFile:setFloat(key .. "#vehicleCollateralPercentage", self.vehicleCollateralPercentage)
    xmlFile:setFloat(key .. "#farmlandCollateralPercentage", self.farmlandCollateralPercentage)
    xmlFile:setInt(key .. "#defaultPaymentTermYears", self.defaultPaymentTermYears)
    xmlFile:setFloat(key .. "#lateFeeAmount", self.lateFeeAmount)
    xmlFile:setInt(key .. "#gracePeriodDays", self.gracePeriodDays)
    xmlFile:setBool(key .. "#enableNotifications", self.enableNotifications)
    xmlFile:setBool(key .. "#showHudOverlay", self.showHudOverlay)
end

function LOC_properties:loadFromXML(xmlFile, key)
    self.calculationMode = xmlFile:getString(key .. "#calculationMode", "realistic")
    self.baseInterestRate = xmlFile:getFloat(key .. "#baseInterestRate", 6.0)
    self.rateFluctuationRange = xmlFile:getFloat(key .. "#rateFluctuationRange", 2.0)
    self.minInterestRate = xmlFile:getFloat(key .. "#minInterestRate", 1.0)
    self.maxInterestRate = xmlFile:getFloat(key .. "#maxInterestRate", 15.0)
    self.dynamicInterestRate = xmlFile:getBool(key .. "#dynamicInterestRate", true)
    self.fixedCreditLimit = xmlFile:getFloat(key .. "#fixedCreditLimit", 5000000)
    self.maxCreditToCollateralRatio = xmlFile:getFloat(key .. "#maxCreditToCollateralRatio", 80.0)
    self.vehicleCollateralPercentage = xmlFile:getFloat(key .. "#vehicleCollateralPercentage", 70.0)
    self.farmlandCollateralPercentage = xmlFile:getFloat(key .. "#farmlandCollateralPercentage", 80.0)
    self.defaultPaymentTermYears = xmlFile:getInt(key .. "#defaultPaymentTermYears", 10)
    self.lateFeeAmount = xmlFile:getFloat(key .. "#lateFeeAmount", 50.0)
    self.gracePeriodDays = xmlFile:getInt(key .. "#gracePeriodDays", 7)
    self.enableNotifications = xmlFile:getBool(key .. "#enableNotifications", true)
    self.showHudOverlay = xmlFile:getBool(key .. "#showHudOverlay", true)
end
