-- Line of Credit Manager
-- Author: sockswayup
-- Version: 1.0.4.0
-- Based on Enhanced Loan System pattern

LOC_manager = {}
LOC_manager.creditLines = {}

local LOC_manager_mt = Class(LOC_manager)

function LOC_manager.new(customMt)
    local self = setmetatable({}, customMt or LOC_manager_mt)
    
    self.creditLines = {}
    
    -- Initialize default properties for client compatibility
    -- These will be overridden by loadMapData when config is loaded
    self.properties = {
        calculationMode = "realistic",
        baseInterestRate = 6.0,
        rateFluctuationRange = 2.0,
        minInterestRate = 1.0,
        maxInterestRate = 15.0,
        dynamicInterestRate = true,
        defaultCreditLimit = 1000000,
        fixedCreditLimit = 5000000,
        maxCreditToCollateralRatio = 80.0,
        vehicleCollateralPercentage = 70.0,
        farmlandCollateralPercentage = 60.0,
        defaultPaymentTermYears = 10,
        minPaymentTermYears = 1,
        maxPaymentTermYears = 30,
        lateFeeAmount = 50.0,
        gracePeriodDays = 7,
        enableNotifications = true,
        showHudOverlay = true,
        
        -- Add required methods
        saveToXML = function(props, xmlFile, key)
            xmlFile:setString(key .. "#calculationMode", props.calculationMode)
            xmlFile:setFloat(key .. "#baseInterestRate", props.baseInterestRate)
            xmlFile:setFloat(key .. "#rateFluctuationRange", props.rateFluctuationRange)
            xmlFile:setFloat(key .. "#minInterestRate", props.minInterestRate)
            xmlFile:setFloat(key .. "#maxInterestRate", props.maxInterestRate)
            xmlFile:setBool(key .. "#dynamicInterestRate", props.dynamicInterestRate)
            xmlFile:setFloat(key .. "#fixedCreditLimit", props.fixedCreditLimit)
            xmlFile:setFloat(key .. "#maxCreditToCollateralRatio", props.maxCreditToCollateralRatio)
            xmlFile:setFloat(key .. "#vehicleCollateralPercentage", props.vehicleCollateralPercentage)
            xmlFile:setFloat(key .. "#farmlandCollateralPercentage", props.farmlandCollateralPercentage)
            xmlFile:setInt(key .. "#defaultPaymentTermYears", props.defaultPaymentTermYears)
            xmlFile:setFloat(key .. "#lateFeeAmount", props.lateFeeAmount)
            xmlFile:setInt(key .. "#gracePeriodDays", props.gracePeriodDays)
            xmlFile:setBool(key .. "#enableNotifications", props.enableNotifications)
            xmlFile:setBool(key .. "#showHudOverlay", props.showHudOverlay)
        end,
        
        loadFromXML = function(props, xmlFile, key)
            props.calculationMode = xmlFile:getString(key .. "#calculationMode", "realistic")
            props.baseInterestRate = xmlFile:getFloat(key .. "#baseInterestRate", 6.0)
            props.rateFluctuationRange = xmlFile:getFloat(key .. "#rateFluctuationRange", 2.0)
            props.minInterestRate = xmlFile:getFloat(key .. "#minInterestRate", 1.0)
            props.maxInterestRate = xmlFile:getFloat(key .. "#maxInterestRate", 15.0)
            props.dynamicInterestRate = xmlFile:getBool(key .. "#dynamicInterestRate", true)
            props.fixedCreditLimit = xmlFile:getFloat(key .. "#fixedCreditLimit", 5000000)
            props.maxCreditToCollateralRatio = xmlFile:getFloat(key .. "#maxCreditToCollateralRatio", 80.0)
            props.vehicleCollateralPercentage = xmlFile:getFloat(key .. "#vehicleCollateralPercentage", 70.0)
            props.farmlandCollateralPercentage = xmlFile:getFloat(key .. "#farmlandCollateralPercentage", 60.0)
            props.defaultPaymentTermYears = xmlFile:getInt(key .. "#defaultPaymentTermYears", 10)
            props.lateFeeAmount = xmlFile:getFloat(key .. "#lateFeeAmount", 50.0)
            props.gracePeriodDays = xmlFile:getInt(key .. "#gracePeriodDays", 7)
            props.enableNotifications = xmlFile:getBool(key .. "#enableNotifications", true)
            props.showHudOverlay = xmlFile:getBool(key .. "#showHudOverlay", true)
        end
    }
    
    return self
end

function LOC_manager:loadMap()
    g_messageCenter:subscribe(MessageType.PERIOD_CHANGED, self.onPeriodChanged, self)
    g_messageCenter:subscribe(MessageType.YEAR_CHANGED, self.onYearChanged, self)
    g_messageCenter:subscribe(MessageType.DAY_CHANGED, self.onDayChanged, self)
    
    -- Set up periodic sync for multiplayer (simpler approach)
    if g_currentMission:getIsServer() then
        print("[Line of Credit] Server initialized - will use delayed sync approach")
    end
end

function LOC_manager:loadMapData(xmlFile)
    -- Load configuration and override default properties
    local tempProperties = LOC_properties.new(g_currentMission:getIsServer(), g_currentMission:getIsClient())
    tempProperties:loadFromConfig()
    
    -- Override the default properties with loaded config values
    g_loc_manager.properties.calculationMode = tempProperties.calculationMode
    g_loc_manager.properties.baseInterestRate = tempProperties.baseInterestRate
    g_loc_manager.properties.rateFluctuationRange = tempProperties.rateFluctuationRange
    g_loc_manager.properties.minInterestRate = tempProperties.minInterestRate
    g_loc_manager.properties.maxInterestRate = tempProperties.maxInterestRate
    g_loc_manager.properties.dynamicInterestRate = tempProperties.dynamicInterestRate
    g_loc_manager.properties.defaultCreditLimit = tempProperties.defaultCreditLimit
    g_loc_manager.properties.fixedCreditLimit = tempProperties.fixedCreditLimit
    g_loc_manager.properties.maxCreditToCollateralRatio = tempProperties.maxCreditToCollateralRatio
    g_loc_manager.properties.vehicleCollateralPercentage = tempProperties.vehicleCollateralPercentage
    g_loc_manager.properties.farmlandCollateralPercentage = tempProperties.farmlandCollateralPercentage
    g_loc_manager.properties.defaultPaymentTermYears = tempProperties.defaultPaymentTermYears
    g_loc_manager.properties.minPaymentTermYears = tempProperties.minPaymentTermYears
    g_loc_manager.properties.maxPaymentTermYears = tempProperties.maxPaymentTermYears
    g_loc_manager.properties.lateFeeAmount = tempProperties.lateFeeAmount
    g_loc_manager.properties.gracePeriodDays = tempProperties.gracePeriodDays
    g_loc_manager.properties.enableNotifications = tempProperties.enableNotifications
    g_loc_manager.properties.showHudOverlay = tempProperties.showHudOverlay
    
    print("[Line of Credit] Properties loaded successfully - Mode: " .. g_loc_manager.properties.calculationMode)
    
    -- Server-specific initialization can go here if needed
    if g_currentMission:getIsServer() then
        -- Server-only logic if any
    end
end

function LOC_manager:onPeriodChanged()
    -- Handle monthly payments and interest rate changes
    -- Process in single player or when we're the server
    if g_currentMission:getIsServer() or g_currentMission.isMasterUser then
        local environment = g_currentMission.environment
        if environment then
        end
        self:processMonthlyUpdates()
    end
end

function LOC_manager:onYearChanged()
    -- Handle yearly updates if needed
end

function LOC_manager:onDayChanged()
    -- Log current day for testing
    if g_currentMission and (g_currentMission:getIsServer() or g_currentMission.isMasterUser) then
        local environment = g_currentMission.environment
        if environment then
            -- Check if we have any active credit lines
            local hasActiveCredit = false
            for farmId, creditLines in pairs(self.creditLines) do
                for _, creditLine in pairs(creditLines) do
                    if not creditLine.closed and creditLine.currentBalance > 0 then
                        hasActiveCredit = true
                        break
                    end
                end
            end
            
            if hasActiveCredit then
            end
        end
    end
end


function LOC_manager:processMonthlyUpdates()
    
    for farmId, creditLines in pairs(self.creditLines) do
        for _, creditLine in pairs(creditLines) do
            if not creditLine.closed and creditLine.currentBalance > 0 then
                
                -- Check for approaching term limit
                local monthsRemaining = creditLine:getRemainingTerm()
                if monthsRemaining == 12 then
                    -- One year warning
                    if g_currentMission.hud then
                        g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_INFO,
                            string.format("Line of Credit: 12 months remaining - Balance: %s",
                                g_i18n:formatMoney(creditLine.currentBalance)))
                    end
                elseif monthsRemaining == 6 then
                    -- 6 months warning
                    if g_currentMission.hud then
                        g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL,
                            string.format("Line of Credit: Only 6 months remaining - Balance: %s",
                                g_i18n:formatMoney(creditLine.currentBalance)))
                    end
                elseif monthsRemaining == 1 then
                    -- Final warning
                    if g_currentMission.hud then
                        g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL,
                            string.format("Line of Credit: FINAL MONTH - Full balance of %s due next month!",
                                g_i18n:formatMoney(creditLine.currentBalance)))
                    end
                elseif monthsRemaining <= 0 and creditLine.currentBalance > 0 then
                    -- Balance is due
                    if g_currentMission.hud then
                        g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL,
                            string.format("Line of Credit EXPIRED: Full balance of %s is due NOW!",
                                g_i18n:formatMoney(creditLine.currentBalance)))
                    end
                end
                
                -- Update interest rate with fluctuation
                self:updateInterestRate(creditLine)
                
                -- Process monthly payment if due
                self:processMonthlyPayment(creditLine)
            end
        end
    end
    -- Save changes to XML after processing all monthly payments
    if g_currentMission:getIsServer() or g_currentMission.isMasterUser then
        -- Debug: show manager state before save
        local totalLines = 0
        for farmId, lines in pairs(self.creditLines) do
            totalLines = totalLines + #lines
            local farmIdStr = tostring(farmId)
            print("[Line of Credit] Before save: Farm " .. farmIdStr .. " has " .. #lines .. " credit lines")
        end
        print("[Line of Credit] Before save: Manager has " .. totalLines .. " total credit lines")
        LOC_manager.saveToXMLFile(g_currentMission.missionInfo)
    end
end

function LOC_manager:updateInterestRate(creditLine)
    if self.properties.dynamicInterestRate then
        local fluctuation = (math.random() - 0.5) * 2 * self.properties.rateFluctuationRange / 100
        creditLine.currentInterestRate = math.max(
            self.properties.minInterestRate / 100,
            math.min(self.properties.maxInterestRate / 100, 
                creditLine.baseInterestRate + fluctuation)
        )
        
    end
end

function LOC_manager:processMonthlyPayment(creditLine)
    -- Check for auto-close FIRST, before payment processing
    local remainingTerm = creditLine:getRemainingTerm()
    
    if creditLine.currentBalance <= 0.01 and remainingTerm <= 0 and not creditLine.closed then
        local success, message = self:closeCreditLine(creditLine)
        if success then
            if g_currentMission.hud then
                g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_INFO,
                    string.format("Credit line automatically closed - Term completed (%d months)",
                        creditLine.paymentTerm))
            end
            return -- Exit early if closed
        else
        end
    end
    
    -- Now check if any payment is still needed
    local remainingPayment = creditLine:getRemainingMonthlyPayment()
    
    
    -- Only process if there's remaining payment due
    if remainingPayment > 0 and creditLine.currentBalance > 0 then
        local farm = g_farmManager:getFarmById(creditLine.farmId)
        if farm and farm.money >= remainingPayment then
            -- Calculate interest and principal portions for the remaining payment
            local monthlyInterestRate = creditLine.currentInterestRate / 12
            local interestPortion = creditLine.currentBalance * monthlyInterestRate
            local principalPortion = remainingPayment - interestPortion
            
            -- Ensure we don't overpay
            if principalPortion > creditLine.currentBalance then
                principalPortion = creditLine.currentBalance
                remainingPayment = creditLine.currentBalance + interestPortion
            end
            
            -- Apply payment
            creditLine.currentBalance = creditLine.currentBalance - principalPortion
            creditLine.totalInterestPaid = creditLine.totalInterestPaid + interestPortion
            creditLine.monthlyPaymentSatisfied = creditLine.monthlyPaymentSatisfied + remainingPayment
            
            -- Remove money from farm
            -- Use OTHER for interest and LOAN for principal to track separately
            if interestPortion > 0 then
                -- Interest portion shows as other expense
                self:addRemoveMoney(-interestPortion, creditLine.farmId, MoneyType.OTHER, 
                    string.format("Line of Credit Interest (%.2f%%)", creditLine.currentInterestRate * 100))
            end
            if principalPortion > 0 then
                -- Principal portion shows as loan payment (using OTHER for now)
                self:addRemoveMoney(-principalPortion, creditLine.farmId, MoneyType.OTHER, 
                    "Line of Credit Principal Payment")
            end
            
            -- Update credit line
            creditLine:recalculateMonthlyPayment()
            
            -- Sync changes to clients (like ELS does)
            creditLine:raiseDirtyFlags(creditLine.creditLineDirtyFlag)
            creditLine:raiseActive()
                        
            -- Auto-close if term expired and balance is zero
            if creditLine.currentBalance <= 0 and creditLine:getRemainingTerm() <= 0 then
                self:closeCreditLine(creditLine)
                if g_currentMission.hud then
                    g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_INFO,
                        string.format("Credit line automatically closed - Term expired (%d months used)",
                            creditLine.paymentTerm))
                end
            end
        else
            -- Apply late fee only if payment wasn't already made manually
            if creditLine.monthlyPaymentSatisfied <= 0 then
                creditLine.currentBalance = creditLine.currentBalance + self.properties.lateFeeAmount
            end
        end
    end
    -- Decrement term after processing payment even if min was hit before sleeping.
    creditLine.paymentsMade = creditLine.paymentsMade + 1
   
    -- Reset for next month AFTER processing
    creditLine:resetMonthlyPayment()
end

function LOC_manager:currentCreditLines(farmId)
    local currentLines = {}
    local allFarmLines = self.creditLines[farmId] or {}
    
    for _, creditLine in pairs(allFarmLines) do
        if not creditLine.closed then
            table.insert(currentLines, creditLine)
        end
    end
    
    return currentLines
end

function LOC_manager:addCreditLine(creditLine)
    print("[Line of Credit] addCreditLine called for farm " .. creditLine.farmId .. " with limit " .. creditLine.creditLimit)
    -- Register the credit line (this handles adding to manager)
    creditLine:register()
    -- No need to manually add again since register() already does this
    
    -- Debug: show current state after registration
    local totalLines = 0
    for farmId, lines in pairs(self.creditLines) do
        totalLines = totalLines + #lines
    end
    print("[Line of Credit] After registration, manager has " .. totalLines .. " total credit lines across all farms")
    
    -- DEBUG: Force an immediate save to test if credit line persists
    if g_currentMission:getIsServer() then
        print("[Line of Credit] DEBUG: Forcing immediate save after credit line creation")
        LOC_manager.saveToXMLFile(g_currentMission.missionInfo)
    end
end

function LOC_manager:closeCreditLine(creditLine)
    if creditLine.currentBalance > 0 then
        return false, "Cannot close credit line with outstanding balance"
    end
    
    creditLine.closed = true
    creditLine.closedDate = g_currentMission.environment.currentDay or 0
    
    -- Sync changes to all clients (like ELS does)
    creditLine:raiseDirtyFlags(creditLine.creditLineDirtyFlag)
    creditLine:raiseActive()
    print("[Line of Credit] Credit line closed and synced for farm " .. creditLine.farmId)
        
    return true, "Credit line closed successfully"
end

function LOC_manager:drawFunds(creditLine, amount)
    if amount <= 0 then
        return false, "Invalid amount"
    end
    
    local availableCredit = creditLine:getAvailableCredit()
    if amount > availableCredit then
        return false, string.format("Insufficient credit. Available: $%.2f", availableCredit)
    end
    
    -- Update balance
    creditLine.currentBalance = creditLine.currentBalance + amount
    creditLine.totalDrawn = creditLine.totalDrawn + amount
    
    -- Track when this draw happened for grace period
    if g_currentMission and g_currentMission.environment then
        creditLine.lastDrawDate = g_currentMission.environment.currentDay or 0
        creditLine.lastDrawPeriod = g_currentMission.environment.currentPeriod or 0
    end
    
    -- Add money to farm - show as loan income in finances (using OTHER for now)
    self:addRemoveMoney(amount, creditLine.farmId, MoneyType.OTHER, "Line of Credit Draw")
    
    -- Recalculate monthly payment
    creditLine:recalculateMonthlyPayment()
    
    -- Sync changes to all clients (like ELS does)
    creditLine:raiseDirtyFlags(creditLine.creditLineDirtyFlag)
    creditLine:raiseActive()
    print("[Line of Credit] Credit line updated and synced - new balance: " .. creditLine.currentBalance)
    
    return true, "Funds drawn successfully"
end

function LOC_manager:makePayment(creditLine, amount)
    if amount <= 0 then
        return false, "Invalid payment amount"
    end
    
    if creditLine.currentBalance <= 0 then
        return false, "No balance to pay"
    end
    
    local farmId = creditLine.farmId
    if farmId == 0 then
        farmId = g_currentMission:getFarmId()
        print("[Line of Credit] Correcting farmId from 0 to " .. farmId)
    end
    
    local farm = g_farmManager:getFarmById(farmId)
    print(string.format("[Line of Credit] Payment check: farmId=%s, farm=%s, farmMoney=%.2f, paymentAmount=%.2f", 
        tostring(farmId), tostring(farm ~= nil), farm and farm.money or 0, amount))
    
    if not farm or farm.money < amount then
        return false, "Insufficient funds"
    end
    
    -- Check if this is a payoff amount
    local payoffAmount = creditLine:getPayoffAmount()
    local isPayoff = math.abs(amount - payoffAmount) < 1  -- Within $1 of payoff amount
    
    -- First, determine how much goes to monthly obligation
    local remainingMonthlyPayment = creditLine:getRemainingMonthlyPayment()
    local towardMonthly = math.min(amount, remainingMonthlyPayment)
    local extraPayment = amount - towardMonthly
    
    -- Check if we're within grace period (same month as last draw)
    local isGracePeriod = false
    if g_currentMission and g_currentMission.environment then
        local currentPeriod = g_currentMission.environment.currentPeriod or 0
        if creditLine.lastDrawPeriod and creditLine.lastDrawPeriod == currentPeriod then
            isGracePeriod = true
        end
    end
    
    -- Calculate payment allocation
    local monthlyInterestRate = creditLine.currentInterestRate / 12
    local interestPortion = creditLine.currentBalance * monthlyInterestRate
    local principalPortion = 0
    
    -- Check if this is a full payoff within grace period
    if isGracePeriod and amount >= creditLine.currentBalance then
        -- Grace period full payoff - no interest charged
        principalPortion = creditLine.currentBalance
        interestPortion = 0
        amount = creditLine.currentBalance  -- Adjust amount to exact balance
    elseif isPayoff then
        -- For normal payoff, apply exactly what's needed to zero the balance
        principalPortion = creditLine.currentBalance
        interestPortion = amount - principalPortion
    else
        -- Normal payment: interest first, then principal
        interestPortion = math.min(amount, interestPortion)
        principalPortion = math.min(amount - interestPortion, creditLine.currentBalance)
    end
    
    if principalPortion < 0 then
        principalPortion = 0
        interestPortion = amount
    end
    
    -- Apply payment
    creditLine.currentBalance = creditLine.currentBalance - principalPortion
    creditLine.totalInterestPaid = creditLine.totalInterestPaid + interestPortion
    
    -- Track payment toward monthly obligation
    creditLine.monthlyPaymentSatisfied = creditLine.monthlyPaymentSatisfied + towardMonthly
    
    -- Only count as extra payment if it's beyond monthly obligation
    if extraPayment > 0 then
        creditLine.extraPaymentsMade = creditLine.extraPaymentsMade + 1
    end
    
    -- Remove money from farm - split for proper tracking
    if interestPortion > 0 then
        -- Interest shows as other expense
        self:addRemoveMoney(-interestPortion, farmId, MoneyType.OTHER, 
            string.format("Line of Credit Interest (%.2f%%)", creditLine.currentInterestRate * 100))
    end
    if principalPortion > 0 then
        -- Principal shows as loan payment (using OTHER for now)
        self:addRemoveMoney(-principalPortion, farmId, MoneyType.OTHER, 
            "Line of Credit Extra Payment")
    end
    
    -- Recalculate monthly payment
    creditLine:recalculateMonthlyPayment()
    
    -- Sync changes to all clients (like ELS does)
    creditLine:raiseDirtyFlags(creditLine.creditLineDirtyFlag)
    creditLine:raiseActive()
    print("[Line of Credit] Credit line payment processed and synced - new balance: " .. creditLine.currentBalance)
        
    return true, "Payment applied successfully"
end

function LOC_manager:addRemoveMoney(amount, farmId, moneyType, description)
    if moneyType == nil then
        moneyType = MoneyType.OTHER
    end
    
    if g_currentMission:getIsServer() or g_currentMission.isMasterUser then
        print(string.format("[Line of Credit] Adding $%.2f to farm %d (type: %s)", amount, farmId, tostring(moneyType)))
        
        -- Add to money change history for finance screen
        -- The 'true, true' parameters make it show in finance stats and history
        g_currentMission:addMoneyChange(amount, farmId, moneyType, true, true)
        
        -- Update farm balance
        local farm = g_farmManager:getFarmById(farmId)
        if farm ~= nil then
            local oldBalance = farm.money
            farm:changeBalance(amount, moneyType)
            print(string.format("[Line of Credit] Farm balance changed from $%.2f to $%.2f", oldBalance, farm.money))
        else
            print("[Line of Credit] Error: Could not find farm with ID " .. tostring(farmId))
        end
    else
        g_client:getServerConnection():sendEvent(LOC_addRemoveMoneyEvent.new(amount, farmId, moneyType, description))
    end
end

function LOC_manager:maxCreditLimitForFarm(farmId)
    local farm = g_farmManager:getFarmById(farmId)
    if not farm then
        print("[Line of Credit] No farm found for ID: " .. tostring(farmId))
        return 0
    end
    
    if not self.properties then
        print("[Line of Credit] Properties not initialized")
        return 0
    end
    
    local existingCredit = self:calculateTotalCreditUsed(farmId)
    local mode = self.properties.calculationMode or "default"
    
    print("[Line of Credit] Credit calculation for farm " .. farmId .. ": mode=" .. mode)
    
    if mode == "sandbox" or mode == "unlimited" then
        -- Sandbox mode - return a very high number (10 million minus existing credit)
        local sandboxCredit = 10000000 - existingCredit
        print("[Line of Credit] Sandbox mode: returning " .. sandboxCredit)
        return sandboxCredit
    else
        -- Default mode - calculate based on collateral
        local vehicleValue = self:calculateVehicleValue(farmId) * (self.properties.vehicleCollateralPercentage / 100)
        local farmlandValue = self:calculateFarmlandValue(farmId) * (self.properties.farmlandCollateralPercentage / 100)
        
        local totalCollateral = farm.money + vehicleValue + farmlandValue
        local maxCredit = totalCollateral * (self.properties.maxCreditToCollateralRatio / 100) - existingCredit
        
        print(string.format("[Line of Credit] Collateral calculation: money=%d, vehicles=%d, farmland=%d, total=%d, maxCredit=%d", 
            farm.money, vehicleValue, farmlandValue, totalCollateral, maxCredit))
                
        return math.max(0, maxCredit)
    end
end

function LOC_manager:calculateVehicleValue(farmId)
    local amount = 0
    
    for _, vehicle in ipairs(g_currentMission.vehicleSystem.vehicles) do
        if vehicle.ownerFarmId == farmId and vehicle.propertyState == VehiclePropertyState.OWNED and vehicle.getSellPrice ~= nil then
            amount = amount + vehicle:getSellPrice()
        end
    end
    
    return amount
end

function LOC_manager:calculateFarmlandValue(farmId)
    local amount = 0
    
    for _, farmland in pairs(g_farmlandManager:getFarmlands()) do
        if g_farmlandManager:getFarmlandOwner(farmland.id) == farmId then
            amount = amount + farmland.price
        end
    end
    
    return amount
end

function LOC_manager:calculateTotalCreditUsed(farmId)
    local amount = 0
    local creditLines = self:currentCreditLines(farmId)
    
    for _, creditLine in pairs(creditLines) do
        amount = amount + creditLine.currentBalance
    end
    
    return amount
end

function LOC_manager.loadFromXMLFile()
    local savegameDir = g_currentMission.missionInfo.savegameDirectory
    if not savegameDir then
        print("[Line of Credit] No savegame directory found")
        return
    end
    
    local xmlFilename = savegameDir .. "/lineOfCredit.xml"
    local key = "lineOfCredit"
    local xmlFile = XMLFile.loadIfExists("LineOfCreditSaveGame", xmlFilename, key)
    
    if not xmlFile then
        print("[Line of Credit] No saved credit data found")
        return
    end
    
    -- Load properties
    g_loc_manager.properties:loadFromXML(xmlFile, key .. ".properties")
    
    -- Load credit lines for each farm using ELS pattern
    local farmIndex = 0
    while true do
        local farmKey = string.format(key .. ".creditLines.farm(%d)", farmIndex)
        
        if not xmlFile:hasProperty(farmKey) then
            break
        end
        
        local farmId = xmlFile:getInt(farmKey .. "#id")
        if farmId then
            local currentFarmCreditLines = {}
            local lineIndex = 0
            
            while true do
                local lineKey = string.format(farmKey .. ".creditLine(%d)", lineIndex)
                
                if not xmlFile:hasProperty(lineKey) then
                    break
                end
                
                local creditLine = LOC_creditLine.new(g_currentMission:getIsServer(), g_currentMission:getIsClient())
                if not creditLine:loadFromXMLFile(xmlFile, lineKey) then
                    creditLine:delete()
                else
                    creditLine:register()  -- This should sync to clients if on server
                    table.insert(currentFarmCreditLines, creditLine)
                    local farmIdStr = tostring(creditLine.farmId or "unknown")
                    local balanceStr = tostring(creditLine.currentBalance or 0)
                    print("[Line of Credit] Loaded credit line for farm " .. farmIdStr .. " with balance " .. balanceStr)
                end
                
                lineIndex = lineIndex + 1
            end
            
            g_loc_manager.creditLines[farmId] = currentFarmCreditLines
        end
        
        farmIndex = farmIndex + 1
    end
    
    xmlFile:delete()
    
    print("[Line of Credit] Finished loading credit lines from XML")
end

-- Sync all existing credit lines from server to clients
function LOC_manager.syncAllCreditLinesToClients()
    if not g_currentMission:getIsServer() then
        print("[Line of Credit] syncAllCreditLinesToClients called on client - ignoring")
        return
    end
    
    print("[Line of Credit] Server syncing ALL credit lines to clients using Object system")
    local totalSynced = 0
    for farmId, creditLines in pairs(g_loc_manager.creditLines) do
        for _, creditLine in ipairs(creditLines) do
            local farmIdStr = tostring(farmId)
            local balanceStr = tostring(creditLine.currentBalance or 0)
            print("[Line of Credit] Server syncing credit line to clients for farm " .. farmIdStr .. " (balance: " .. balanceStr .. ")")
            creditLine:raiseActive()  -- Use Object system like ELS
            totalSynced = totalSynced + 1
        end
    end
    print("[Line of Credit] Total credit lines synced using Object system: " .. totalSynced)
end

-- Sync credit lines for a specific farm
function LOC_manager:syncCreditLinesForFarm(farmId)
    if not g_currentMission:getIsServer() then
        return
    end
    
    local creditLines = self.creditLines[farmId]
    if not creditLines or #creditLines == 0 then
        local farmIdStr = tostring(farmId)
        print("[Line of Credit] No credit lines found for farm " .. farmIdStr .. " to sync")
        return
    end
    
    local farmIdStr = tostring(farmId)
    print("[Line of Credit] Syncing " .. #creditLines .. " credit lines for farm " .. farmIdStr)
    for _, creditLine in ipairs(creditLines) do
        local balanceStr = tostring(creditLine.currentBalance or 0)
        print("[Line of Credit] Syncing credit line with balance " .. balanceStr)
        creditLine:raiseActive()  -- Use Object system
    end
end

function LOC_manager.saveToXMLFile(missionInfo)
    print("[Line of Credit] saveToXMLFile called!")
    local savegameDir = g_currentMission.missionInfo.savegameDirectory
    if not savegameDir then
        print("[Line of Credit] No savegame directory found for saving")
        return
    end
    
    local xmlFilename = savegameDir .. "/lineOfCredit.xml"
    local key = "lineOfCredit"
    print("[Line of Credit] Attempting to save to: " .. xmlFilename)
    local xmlFile = XMLFile.create("LineOfCreditSaveGame", xmlFilename, key)
    
    if not xmlFile then
        print("[Line of Credit] Failed to create XML file for saving")
        return
    end
    
    -- Save properties
    g_loc_manager.properties:saveToXML(xmlFile, key .. ".properties")
    
    -- Save credit lines for each farm using ELS pattern
    print("[Line of Credit] Saving credit lines...")
    local totalCreditLines = 0
    for farmId, creditLines in pairs(g_loc_manager.creditLines) do
        totalCreditLines = totalCreditLines + #creditLines
    end
    print("[Line of Credit] Total credit lines to save: " .. totalCreditLines)
    
    local farmIndex = 0
    for farmId, creditLines in pairs(g_loc_manager.creditLines) do
        if #creditLines > 0 then
            print("[Line of Credit] Saving " .. #creditLines .. " credit lines for farm " .. farmId)
            local farmKey = string.format(key .. ".creditLines.farm(%d)", farmIndex)
            xmlFile:setInt(farmKey .. "#id", farmId)
            
            for lineIndex, creditLine in ipairs(creditLines) do
                local lineKey = string.format("%s.creditLine(%d)", farmKey, lineIndex - 1)
                local balanceStr = tostring(creditLine.currentBalance or 0)
                print("[Line of Credit] Saving credit line " .. lineIndex .. " with balance: " .. balanceStr)
                creditLine:saveToXMLFile(xmlFile, lineKey)
            end
            
            farmIndex = farmIndex + 1
        end
    end
    
    print("[Line of Credit] Calling xmlFile:save()")
    xmlFile:save()
    xmlFile:delete()
    print("[Line of Credit] Save completed successfully!")
end

-- Create global instance
g_loc_manager = LOC_manager.new()