-- Line of Credit - Individual Credit Line
-- Author: sockswayup
-- Version: 1.0.4.0
-- Simplified version without network sync

LOC_creditLine = {}
local LOC_creditLine_mt = Class(LOC_creditLine, Object)

InitObjectClass(LOC_creditLine, "LOC_creditLine")

function LOC_creditLine.new(isServer, isClient, customMt)
    local self = Object.new(isServer, isClient, customMt or LOC_creditLine_mt)
    
    self.creditLineDirtyFlag = self:getNextDirtyFlag()
    
    self.farmId = g_currentMission and g_currentMission:getFarmId() or 1
    self.creditLimit = 0
    self.currentBalance = 0
    self.baseInterestRate = 0.06 -- 6% default
    self.currentInterestRate = 0.06
    self.paymentTerm = 120 -- 120 months (10 years) maximum term
    self.monthlyPayment = 0
    self.totalDrawn = 0
    self.totalInterestPaid = 0
    self.paymentsMade = 0
    self.extraPaymentsMade = 0
    self.monthlyPaymentSatisfied = 0 -- Track amount paid toward current month's obligation
    self.lastMonthlyPaymentReset = 0 -- Track when monthly payment was last reset
    self.approved = false
    self.closed = false
    self.closedDate = 0
    self.applicationDate = 0
    self.lastPaymentDate = 0
    self.lastDrawDate = 0 -- Track when funds were last drawn
    self.lastDrawPeriod = 0 -- Track the period (month) of last draw
    
    return self
end

function LOC_creditLine:init(farmId, creditLimit, interestRate, paymentTermYears, isDynamic)
    self.farmId = farmId
    self.creditLimit = creditLimit
    self.baseInterestRate = interestRate / 100 -- Convert percentage to decimal
    self.currentInterestRate = self.baseInterestRate
    self.paymentTerm = paymentTermYears * 12 -- Convert years to months
    self.approved = true
    self.applicationDate = g_currentMission.environment.currentDay or 0
    
end

function LOC_creditLine:getAvailableCredit()
    if not self.approved or self.closed then
        return 0
    end
    return math.max(0, self.creditLimit - self.currentBalance)
end

function LOC_creditLine:recalculateMonthlyPayment()
    
    if self.currentBalance <= 0 then
        self.monthlyPayment = 0
        return
    end
    
    -- Calculate monthly payment using standard loan formula
    -- PMT = P * [r(1+r)^n] / [(1+r)^n - 1]
    local monthlyRate = self.currentInterestRate / 12
    local remainingPayments = self.paymentTerm - self.paymentsMade
    
    -- After 120 months, full balance is due
    if remainingPayments <= 0 then
        self.monthlyPayment = self.currentBalance -- Pay full balance if term expired
        return
    end
    
    if monthlyRate <= 0 then
        self.monthlyPayment = self.currentBalance / remainingPayments -- Simple division if no interest
        return
    end
    
    local factor = math.pow(1 + monthlyRate, remainingPayments)
    local numerator = self.currentBalance * monthlyRate * factor
    local denominator = factor - 1
    
    self.monthlyPayment = numerator / denominator
end

function LOC_creditLine:getUtilizationRate()
    if self.creditLimit <= 0 then
        return 0
    end
    return (self.currentBalance / self.creditLimit) * 100
end

function LOC_creditLine:getRemainingTerm()
    return math.max(0, self.paymentTerm - self.paymentsMade)
end

function LOC_creditLine:getPayoffAmount()
    -- Calculate payoff amount including current interest
    if self.currentBalance <= 0 then
        return 0
    end
    
    -- Check if we're in grace period (same month as draw)
    local isGracePeriod = false
    if g_currentMission and g_currentMission.environment then
        local currentPeriod = g_currentMission.environment.currentPeriod or 0
        if self.lastDrawPeriod and self.lastDrawPeriod == currentPeriod then
            isGracePeriod = true
        end
    end
    
    if isGracePeriod then
        -- Grace period - no interest charged
        return self.currentBalance
    else
        -- Normal payoff - add one month's interest
        local monthlyInterestRate = self.currentInterestRate / 12
        local interestPortion = self.currentBalance * monthlyInterestRate
        local payoffAmount = self.currentBalance + interestPortion
        
        
        -- Return balance plus interest to fully pay off
        return payoffAmount
    end
end

function LOC_creditLine:getRemainingMonthlyPayment()
    -- Get the amount still owed for this month's payment
    if self.monthlyPayment <= 0 then
        return 0
    end
    return math.max(0, self.monthlyPayment - self.monthlyPaymentSatisfied)
end

function LOC_creditLine:resetMonthlyPayment()
    -- Reset the monthly payment tracking (called at start of new month)
    self.monthlyPaymentSatisfied = 0
    if g_currentMission and g_currentMission.environment then
        self.lastMonthlyPaymentReset = g_currentMission.environment.currentDay or 0
    end
end

function LOC_creditLine:register()
    -- Register this credit line in the manager
    if not self.farmId then
        print("[Line of Credit] Error: Cannot register credit line - farmId is nil")
        return
    end
    
    local farmIdStr = tostring(self.farmId)
    local balanceStr = tostring(self.currentBalance or 0)
    
    print("[Line of Credit] Registering credit line for farm " .. farmIdStr .. " with balance " .. balanceStr)
    local farmLoans = g_loc_manager.creditLines[self.farmId] or {}
    print("[Line of Credit] Current farm " .. farmIdStr .. " has " .. #farmLoans .. " existing credit lines")
    
    -- Always add to local manager first
    table.insert(farmLoans, self)
    g_loc_manager.creditLines[self.farmId] = farmLoans
    print("[Line of Credit] Credit line added to manager. Farm " .. farmIdStr .. " now has " .. #farmLoans .. " credit lines")
    
    -- Call Object.register() for network synchronization like ELS
    Object.register(self)
    print("[Line of Credit] Object.register() called - should sync to clients if on server")
end

function LOC_creditLine:delete()
    -- Remove from manager if needed
    if g_loc_manager and g_loc_manager.creditLines and g_loc_manager.creditLines[self.farmId] then
        for i, existingLine in ipairs(g_loc_manager.creditLines[self.farmId]) do
            if existingLine == self then
                table.remove(g_loc_manager.creditLines[self.farmId], i)
                break
            end
        end
    end
end

function LOC_creditLine:saveToXMLFile(xmlFile, key)
    xmlFile:setInt(key .. "#farmId", self.farmId or 0)
    xmlFile:setFloat(key .. "#creditLimit", self.creditLimit)
    xmlFile:setFloat(key .. "#currentBalance", self.currentBalance)
    xmlFile:setFloat(key .. "#baseInterestRate", self.baseInterestRate)
    xmlFile:setFloat(key .. "#currentInterestRate", self.currentInterestRate)
    xmlFile:setInt(key .. "#paymentTerm", self.paymentTerm)
    xmlFile:setFloat(key .. "#monthlyPayment", self.monthlyPayment)
    xmlFile:setFloat(key .. "#totalDrawn", self.totalDrawn)
    xmlFile:setFloat(key .. "#totalInterestPaid", self.totalInterestPaid)
    xmlFile:setInt(key .. "#paymentsMade", self.paymentsMade)
    xmlFile:setInt(key .. "#extraPaymentsMade", self.extraPaymentsMade)
    xmlFile:setFloat(key .. "#monthlyPaymentSatisfied", self.monthlyPaymentSatisfied)
    xmlFile:setInt(key .. "#lastMonthlyPaymentReset", self.lastMonthlyPaymentReset)
    xmlFile:setBool(key .. "#approved", self.approved)
    xmlFile:setBool(key .. "#closed", self.closed)
    xmlFile:setInt(key .. "#closedDate", self.closedDate or 0)
    xmlFile:setInt(key .. "#applicationDate", self.applicationDate or 0)
    xmlFile:setInt(key .. "#lastPaymentDate", self.lastPaymentDate or 0)
    xmlFile:setInt(key .. "#lastDrawDate", self.lastDrawDate or 0)
    xmlFile:setInt(key .. "#lastDrawPeriod", self.lastDrawPeriod or 0)
end

function LOC_creditLine:loadFromXMLFile(xmlFile, key)
    self.farmId = xmlFile:getInt(key .. "#farmId")
    self.creditLimit = xmlFile:getFloat(key .. "#creditLimit")
    self.currentBalance = xmlFile:getFloat(key .. "#currentBalance")
    self.baseInterestRate = xmlFile:getFloat(key .. "#baseInterestRate")
    self.currentInterestRate = xmlFile:getFloat(key .. "#currentInterestRate")
    self.paymentTerm = xmlFile:getInt(key .. "#paymentTerm")
    self.monthlyPayment = xmlFile:getFloat(key .. "#monthlyPayment")
    self.totalDrawn = xmlFile:getFloat(key .. "#totalDrawn")
    self.totalInterestPaid = xmlFile:getFloat(key .. "#totalInterestPaid")
    self.paymentsMade = xmlFile:getInt(key .. "#paymentsMade")
    self.extraPaymentsMade = xmlFile:getInt(key .. "#extraPaymentsMade")
    self.monthlyPaymentSatisfied = xmlFile:getFloat(key .. "#monthlyPaymentSatisfied")
    self.lastMonthlyPaymentReset = xmlFile:getInt(key .. "#lastMonthlyPaymentReset")
    self.approved = xmlFile:getBool(key .. "#approved")
    self.closed = xmlFile:getBool(key .. "#closed")
    self.closedDate = xmlFile:getInt(key .. "#closedDate")
    self.applicationDate = xmlFile:getInt(key .. "#applicationDate")
    self.lastPaymentDate = xmlFile:getInt(key .. "#lastPaymentDate")
    self.lastDrawDate = xmlFile:getInt(key .. "#lastDrawDate")
    self.lastDrawPeriod = xmlFile:getInt(key .. "#lastDrawPeriod")
    
    return true
end

-- Network synchronization methods for multiplayer
function LOC_creditLine:readStream(streamId, connection)
    LOC_creditLine:superClass().readStream(self, streamId, connection)
    
    self.farmId = streamReadInt32(streamId)
    self.creditLimit = streamReadFloat32(streamId)
    self.currentBalance = streamReadFloat32(streamId)
    self.baseInterestRate = streamReadFloat32(streamId)
    self.currentInterestRate = streamReadFloat32(streamId)
    self.paymentTerm = streamReadInt32(streamId)
    self.monthlyPayment = streamReadFloat32(streamId)
    self.totalDrawn = streamReadFloat32(streamId)
    self.totalInterestPaid = streamReadFloat32(streamId)
    self.paymentsMade = streamReadInt32(streamId)
    self.extraPaymentsMade = streamReadInt32(streamId)
    self.monthlyPaymentSatisfied = streamReadFloat32(streamId)
    self.lastMonthlyPaymentReset = streamReadInt32(streamId)
    self.approved = streamReadBool(streamId)
    self.closed = streamReadBool(streamId)
    self.closedDate = streamReadInt32(streamId)
    self.applicationDate = streamReadInt32(streamId)
    self.lastPaymentDate = streamReadInt32(streamId)
    self.lastDrawDate = streamReadInt32(streamId)
    self.lastDrawPeriod = streamReadInt32(streamId)
    
    -- Register in local manager when received via network - like ELS does
    if self.farmId then
        local farmLoans = g_loc_manager.creditLines[self.farmId] or {}
        table.insert(farmLoans, self)
        g_loc_manager.creditLines[self.farmId] = farmLoans
        
        local farmIdStr = tostring(self.farmId)
        local balanceStr = tostring(self.currentBalance or 0)
        print("[Line of Credit] Credit line synchronized for farm " .. farmIdStr .. " - Balance: " .. balanceStr)
    else
        print("[Line of Credit] Error: Cannot sync credit line - farmId is nil")
    end
end

function LOC_creditLine:writeStream(streamId, connection)
    print("[Line of Credit] writeStream called - sending credit line to server")
    LOC_creditLine:superClass().writeStream(self, streamId, connection)
    
    streamWriteInt32(streamId, self.farmId)
    streamWriteFloat32(streamId, self.creditLimit)
    streamWriteFloat32(streamId, self.currentBalance)
    streamWriteFloat32(streamId, self.baseInterestRate)
    streamWriteFloat32(streamId, self.currentInterestRate)
    streamWriteInt32(streamId, self.paymentTerm)
    streamWriteFloat32(streamId, self.monthlyPayment)
    streamWriteFloat32(streamId, self.totalDrawn)
    streamWriteFloat32(streamId, self.totalInterestPaid)
    streamWriteInt32(streamId, self.paymentsMade)
    streamWriteInt32(streamId, self.extraPaymentsMade)
    streamWriteFloat32(streamId, self.monthlyPaymentSatisfied)
    streamWriteInt32(streamId, self.lastMonthlyPaymentReset)
    streamWriteBool(streamId, self.approved)
    streamWriteBool(streamId, self.closed)
    streamWriteInt32(streamId, self.closedDate)
    streamWriteInt32(streamId, self.applicationDate)
    streamWriteInt32(streamId, self.lastPaymentDate)
    streamWriteInt32(streamId, self.lastDrawDate)
    streamWriteInt32(streamId, self.lastDrawPeriod)
    
    local farmIdStr = tostring(self.farmId or "unknown")
    print("[Line of Credit] writeStream completed for farm " .. farmIdStr)
end

function LOC_creditLine:readUpdateStream(streamId, timestamp, connection)
    LOC_creditLine:superClass().readUpdateStream(self, streamId, timestamp, connection)
    
    self.currentBalance = streamReadFloat32(streamId)
    self.monthlyPayment = streamReadFloat32(streamId)
    self.totalDrawn = streamReadFloat32(streamId)
    self.totalInterestPaid = streamReadFloat32(streamId)
    self.paymentsMade = streamReadInt32(streamId)
    self.extraPaymentsMade = streamReadInt32(streamId)
    self.monthlyPaymentSatisfied = streamReadFloat32(streamId)
    self.closed = streamReadBool(streamId)
end

function LOC_creditLine:writeUpdateStream(streamId, connection, dirtyMask)
    LOC_creditLine:superClass().writeUpdateStream(self, streamId, connection, dirtyMask)
    
    streamWriteFloat32(streamId, self.currentBalance)
    streamWriteFloat32(streamId, self.monthlyPayment)
    streamWriteFloat32(streamId, self.totalDrawn)
    streamWriteFloat32(streamId, self.totalInterestPaid)
    streamWriteInt32(streamId, self.paymentsMade)
    streamWriteInt32(streamId, self.extraPaymentsMade)
    streamWriteFloat32(streamId, self.monthlyPaymentSatisfied)
    streamWriteBool(streamId, self.closed)
end
