-- InputHelpPager Mod für FS25
-- Version: 3.5.0 (Strict XML Schema Implementation)

InputHelpPager = {}
InputHelpPager.name = g_currentModName

-- Pagination
InputHelpPager.currentPage = 1
InputHelpPager.totalItems = 0
InputHelpPager.totalPages = 1
InputHelpPager.itemsPerPage = 12

-- Settings & Schema
InputHelpPager.hiddenActions = {}
InputHelpPager.settingsPath = ""
InputHelpPager.xmlSchema = nil

-- Action IDs
InputHelpPager.actionNameUp = "INPUT_HELP_PAGE_UP"
InputHelpPager.actionNameDown = "INPUT_HELP_PAGE_DOWN"

InputHelpPager.origGetEventHelpElements = nil
InputHelpPager.hudHookInstalled = false

-- 1. LOAD
function InputHelpPager:loadMap(name)
    print("InputHelpPager v3.5.0: Loading...")

    local modSettingsDir = getUserProfileAppPath() .. "modSettings/"
    local mySettingsDir = modSettingsDir .. "FS25_InputHelpPager/"
    createFolder(modSettingsDir)
    createFolder(mySettingsDir)
    self.settingsPath = mySettingsDir .. "blockedActions.xml"

    if not self.xmlSchema then
        self.xmlSchema = XMLSchema.new("InputHelpPagerSettings")
        self.xmlSchema:register(XMLValueType.STRING, "blockedActions.action(?)#name", "Internal Action Name")
    end

    self:loadSettings()

    -- Hooks
    PlayerInputComponent.registerGlobalPlayerActionEvents = Utils.appendedFunction(PlayerInputComponent.registerGlobalPlayerActionEvents, InputHelpPager.registerGlobalActionEvents)
    PlayerInputComponent.unregisterActionEvents = Utils.appendedFunction(PlayerInputComponent.unregisterActionEvents, InputHelpPager.unregisterActionEvents)

    -- Console Commands
    addConsoleCommand("ihpNext", "InputHelpPager: Next", "consoleNext", self)
    addConsoleCommand("ihpPrev", "InputHelpPager: Prev", "consolePrev", self)
    addConsoleCommand("ihpDump", "Dump Actions", "consoleDumpActions", self)
    addConsoleCommand("ihpReload", "Reload XML", "loadSettings", self)
end

--function InputHelpPager:deleteMap()
--    if self.xmlSchema then self.xmlSchema:delete() end
--end
function InputHelpPager:mouseEvent(posX, posY, isDown, isUp, button) end
function InputHelpPager:keyEvent(unicode, sym, modifier, isDown) end
function InputHelpPager:draw(dt) end
function InputHelpPager:update(dt) end

-- 3. SETTINGS (MIT SCHEMA)
function InputHelpPager:loadSettings()
    self.hiddenActions = {}
    self.hiddenActions[self.actionNameUp] = true
    self.hiddenActions[self.actionNameDown] = true

    if not fileExists(self.settingsPath) then
        local xmlFile = XMLFile.create("InputHelpSettings", self.settingsPath, "blockedActions", self.xmlSchema)
        if xmlFile then
            xmlFile:setValue("blockedActions.action(0)#name", "EXAMPLE_ACTION_NAME")
            xmlFile:save()
            xmlFile:delete()
            print("InputHelpPager: Created default XML settings.")
        end
        return
    end

    local xmlFile = XMLFile.load("InputHelpSettings", self.settingsPath, self.xmlSchema)
    if xmlFile then
        local i = 0
        while true do
            local key = string.format("blockedActions.action(%d)", i)
            if not xmlFile:hasProperty(key) then break end

            local name = xmlFile:getValue(key .. "#name")
            if name then
                self.hiddenActions[name] = true
                print("InputHelpPager: Hiding '" .. name .. "'")
            end
            i = i + 1
        end
        xmlFile:delete()
        print("InputHelpPager: Settings loaded via Schema.")

        if self.hudDisplay and self.hudDisplay.updateEntries then
            self.hudDisplay:updateEntries(true)
        end
    else
        print("InputHelpPager ERROR: Failed to load XML. Check syntax!")
    end
end

-- 4. DUMP
function InputHelpPager:consoleDumpActions()
    print("----------- F1 MENU DUMP -----------")
    if g_inputDisplayManager then
        local list = self.origGetEventHelpElements(g_inputDisplayManager, g_inputBinding:getComboCommandPressedMask(), false)
        if list then
            for _, item in ipairs(list) do
                print(string.format('    <action name="%s"/>', tostring(item.actionName)))
            end
        end
    end
    print("------------------------------------")
    print("File: " .. self.settingsPath)
end

-- 5. INPUT SYSTEM
function InputHelpPager.registerGlobalActionEvents(playerInputComponent, contextName)
    if playerInputComponent.player ~= nil and playerInputComponent.player.isOwner then
        local self = InputHelpPager
        local currentContextName = g_inputBinding:getContextName()
        local newContextName = contextName or currentContextName

        if currentContextName ~= newContextName then
            g_inputBinding:beginActionEventsModification(newContextName)
        end

        local _, eventIdUp = g_inputBinding:registerActionEvent(self.actionNameUp, self, self.onInputPageUp, false, true, false, true)
        if eventIdUp then
            g_inputBinding:setActionEventTextPriority(eventIdUp, GS_PRIO_HIGH)
            g_inputBinding:setActionEventTextVisibility(eventIdUp, false)
        end

        local _, eventIdDown = g_inputBinding:registerActionEvent(self.actionNameDown, self, self.onInputPageDown, false, true, false, true)
        if eventIdDown then
            g_inputBinding:setActionEventTextPriority(eventIdDown, GS_PRIO_HIGH)
            g_inputBinding:setActionEventTextVisibility(eventIdDown, false)
        end

        if currentContextName ~= newContextName then
            g_inputBinding:beginActionEventsModification(currentContextName)
        end
    end
end

function InputHelpPager.unregisterActionEvents(playerInputComponent)
    if playerInputComponent.player ~= nil and playerInputComponent.player.isOwner then
        g_inputBinding:removeActionEventsByTarget(InputHelpPager)
    end
end

function InputHelpPager:onInputPageUp(actionName, inputValue) self:changePage(-1) end
function InputHelpPager:onInputPageDown(actionName, inputValue) self:changePage(1) end
function InputHelpPager:consoleNext() self:changePage(1) end
function InputHelpPager:consolePrev() self:changePage(-1) end

function InputHelpPager:changePage(direction)
    if self.totalPages <= 1 then return end
    if direction > 0 then
        if self.currentPage < self.totalPages then self.currentPage = self.currentPage + 1 else self.currentPage = 1 end
    else
        if self.currentPage > 1 then self.currentPage = self.currentPage - 1 else self.currentPage = self.totalPages end
    end
    if self.hudDisplay and self.hudDisplay.updateEntries then
        self.hudDisplay:updateEntries(true)
    end
end

-- 6. HUD HOOK & FILTER
function InputHelpPager:installHudHookDynamic()
    if self.hudHookInstalled then return end
    if InputHelpDisplay and InputHelpDisplay.draw then
        local oldDraw = InputHelpDisplay.draw
        InputHelpDisplay.draw = function(displaySelf, offsetX, offsetY)
            self:drawHook(displaySelf, oldDraw, offsetX, offsetY)
        end
        self.hudHookInstalled = true
        print("InputHelpPager: HUD Hook installed.")
    elseif g_currentMission and g_currentMission.hud and g_currentMission.hud.inputHelpDisplay then
        local target = g_currentMission.hud.inputHelpDisplay
        local oldDraw = target.draw
        target.draw = function(displaySelf, offsetX, offsetY)
            self:drawHook(displaySelf, oldDraw, offsetX, offsetY)
        end
        self.hudHookInstalled = true
        print("InputHelpPager: HUD Hook installed.")
    end
end

function InputHelpPager.getEventHelpElementsWrapped(managerSelf, pressedMask, useGamepad)
    local rawList = InputHelpPager.origGetEventHelpElements(managerSelf, pressedMask, useGamepad)

    if rawList == nil or #rawList == 0 then
        InputHelpPager.totalItems = 0
        InputHelpPager.totalPages = 1
        return rawList
    end

    local filteredList = {}
    for _, entry in ipairs(rawList) do
        if not InputHelpPager.hiddenActions[entry.actionName] then
            table.insert(filteredList, entry)
        end
    end

    InputHelpPager.totalItems = #filteredList

    local limit = 12
    if InputHelpDisplay and InputHelpDisplay.MAX_NUM_ELEMENTS then
        limit = InputHelpDisplay.MAX_NUM_ELEMENTS
    end
    InputHelpPager.itemsPerPage = limit
    InputHelpPager.totalPages = math.ceil(InputHelpPager.totalItems / limit)

    if InputHelpPager.totalPages == 0 then InputHelpPager.totalPages = 1 end
    if InputHelpPager.currentPage > InputHelpPager.totalPages then InputHelpPager.currentPage = 1 end
    if InputHelpPager.currentPage < 1 then InputHelpPager.currentPage = 1 end

    local slicedList = {}
    local startIndex = (InputHelpPager.currentPage - 1) * limit + 1
    local endIndex = startIndex + limit - 1

    for i = startIndex, math.min(endIndex, InputHelpPager.totalItems) do
        table.insert(slicedList, filteredList[i])
    end

    return slicedList
end

function InputHelpPager:drawHook(displaySelf, superFunc, offsetX, offsetY)
    self.hudDisplay = displaySelf

    if g_inputDisplayManager then
        self.origGetEventHelpElements = g_inputDisplayManager.getEventHelpElements
        g_inputDisplayManager.getEventHelpElements = self.getEventHelpElementsWrapped
    end

    local status, err = pcall(superFunc, displaySelf, offsetX, offsetY)

    if g_inputDisplayManager and self.origGetEventHelpElements then
        g_inputDisplayManager.getEventHelpElements = self.origGetEventHelpElements
    end

    if not status then return end

    if self.totalPages > 1 and displaySelf:getVisible() then
        self:renderPageInfo(displaySelf)
    end
end

function InputHelpPager:renderPageInfo(displaySelf)
    local posX, posY = displaySelf:getPosition()
    local width = 0.1
    if displaySelf.lineBg and displaySelf.lineBg.width then width = displaySelf.lineBg.width end

    local textSize = displaySelf.textSize or 0.012
    local yOffset = 0.02
    if displaySelf.scalePixelToScreenHeight then yOffset = displaySelf:scalePixelToScreenHeight(20) end

    local headerY = posY
    if displaySelf.comboBg and displaySelf.comboBg.height then
        headerY = posY + (displaySelf.comboBg.height * 0.25)
    else
        headerY = posY + yOffset
    end

    local l10nText = g_i18n:getText("text_INPUT_HELP_PAGER_STATUS")
    if l10nText == "text_INPUT_HELP_PAGER_STATUS" then l10nText = "Page %d / %d" end
    local text = string.format(l10nText, self.currentPage, self.totalPages)

    setTextAlignment(RenderText.ALIGN_RIGHT)
    setTextBold(true)

    local shadowOff = 0.001
    if displaySelf.scalePixelToScreenHeight then shadowOff = displaySelf:scalePixelToScreenHeight(1) end

    setTextColor(0, 0, 0, 1)
    renderText(posX + width - 0.005 + shadowOff, headerY - shadowOff, textSize, text)
    setTextColor(1, 1, 1, 1)
    renderText(posX + width - 0.005, headerY, textSize, text)

    setTextAlignment(RenderText.ALIGN_LEFT)
    setTextBold(false)
end

local function lateUpdate(dt)
    if not InputHelpPager.hudHookInstalled then
        InputHelpPager:installHudHookDynamic()
    end
end
local DummyUpdater = {}
function DummyUpdater:update(dt) lateUpdate(dt) end

addModEventListener(InputHelpPager)
addModEventListener(DummyUpdater)