--[[
Enables you to buy used equipment in the shop

Author:     w33zl / WZL Modding (github.com/w33zl | facebook.com/w33zl)
Version:    1.0.0
Modified:   2025-02-22

Changelog:

]]

BuyUsedEquipment = Mod:init()

BuyUsedEquipment:source("ShopConfigScreenExtension.lua")
BuyUsedEquipment:source("FarmExtension.lua")
BuyUsedEquipment:source("RequestItemEvent.lua")
BuyUsedEquipment:source("NotifySearchCompletedEvent.lua")


BuyUsedEquipment.MIN_SALE_DURATION = 24
BuyUsedEquipment.MAX_SALE_DURATION = BuyUsedEquipment.MIN_SALE_DURATION * 3

BuyUsedEquipment.MIN_GENERATION = 1
BuyUsedEquipment.MAX_GENERATION = 6

BuyUsedEquipment.USE_ALT_PRICE_STRATEGY = true
BuyUsedEquipment.ALT_PRICE_DELTA = 5

local GENERATIONS = {
    {
        maxYear = 0,
        age = { 5, 25 },
        discount = { 0.12, 0.1875 },
        hours = { 2.5, 12.5 },
        damage = { 0.05, 0.25 },
        wear = { 0.045, 0.2375 },
    },
    {
        maxYear = 3,
        age = { 15, 35 },
        discount = { 0.2, 0.3125 },
        hours = { 7.5, 17.5 },
        damage = { 0.15, 0.35 },
        wear = { 0.135, 0.3325 },
    },
    {
        maxYear = 8,
        age = { 25, 45 },
        discount = { 0.28, 0.4375 },
        hours = { 12.5, 22.5 },
        damage = { 0.25, 0.45 },
        wear = { 0.225, 0.4275 },
    },
    {
        maxYear = 15,
        age = { 35, 55 },
        discount = { 0.36, 0.5625 },
        hours = { 17.5, 27.5 },
        damage = { 0.35, 0.55 },
        wear = { 0.315, 0.5225 },
    },
    {
        maxYear = 25,
        age = { 45, 65 },
        discount = { 0.44, 0.6875 },
        hours = { 22.5, 32.5 },
        damage = { 0.45, 0.65 },
        wear = { 0.405, 0.6175 },
    },
    {
        maxYear = 50,
        age = { 55, 75 },
        discount = { 0.52, 0.8125 },
        hours = { 27.5, 37.5 },
        damage = { 0.55, 0.75 },
        wear = { 0.495, 0.7125 },
    },
    {
        maxYear = 100,
        age = { 65, 85 },
        discount = { 0.6, 0.9375 },
        hours = { 32.5, 42.5 },
        damage = { 0.65, 0.85 },
        wear = { 0.585, 0.8075 },
    },
}

local HOURS_PER_MONTH = 24 --HACK: change back to 24?

BuyUsedEquipment.SEARCH_LEVELS = {
    {
        name = g_i18n:getText("searchLevel_normal"),
        duration = 1,
        chance = 0.65, --0.65, --HACK: change back?
        baseFee = 90,
    },
    {
        name = g_i18n:getText("searchLevel_extended"),
        duration = 3,
        chance = 0.80, --0.8, --HACK: change back?
        baseFee = 272,
    },
    {
        name = g_i18n:getText("searchLevel_continuous"),
        duration = 12,
        chance = 0.99, --1.0, --HACK: change back?
        baseFee = 1365,
    },
}

BuyUsedEquipment.MAX_GENERATION = math.min(BuyUsedEquipment.MAX_GENERATION, #GENERATIONS)


function BuyUsedEquipment:requestUsedItem(storeItem, searchLevel)
    local xmlFilename = storeItem.xmlFilename
    Log:debug("requestUsedItem: '%s' '%s'", storeItem.name, xmlFilename)
    RequestItemEvent.requestUsedItem(g_localPlayer.farmId, xmlFilename,searchLevel)
end

function BuyUsedEquipment:calculateFee(price, searchLevel)
    local searchType = self.SEARCH_LEVELS[searchLevel or 1]
    local threshold = 2500 -- Low price threshold
    local dynamicFee = 728
    local factor = math.log10(price/threshold)
    local baseFee = searchType.baseFee
    local fee = baseFee + (math.max(factor, 0) * dynamicFee)
    
    return fee
end


function BuyUsedEquipment:createSearchAssignment(xmlFilename, searchLevel)
    local searchType = self.SEARCH_LEVELS[searchLevel or 1]

    math.random() -- Dry run to improve randomness
    math.random() -- Dry run to improve randomness

    local isSuccess = math.random() <= searchType.chance
    local maxSearchTime = g_currentMission.environment.daysPerPeriod * searchType.duration * HOURS_PER_MONTH
    local searchDuration = math.random(1, maxSearchTime)
    local successTime = isSuccess and math.random(1, searchDuration) or searchDuration + 1

    return {
        ttl = searchDuration,
        tts = successTime,
        filename = xmlFilename,
        level = searchLevel,
    }
end

function BuyUsedEquipment:storeRequestedItem(farmId, xmlFilename, searchLevel)
    local storeItem = g_storeManager:getItemByXMLFilename(xmlFilename)
    local farm = g_farmManager:getFarmById(farmId)

    if farm == nil then
        Log:error("Could not find farm with #%d", farmId)
        return
    end

    FarmExtension.addUsedVehicleSearch(farm, xmlFilename, searchLevel) --farm:addUsedVehicleSearch(xmlFilename)

    local fee = self:calculateFee(storeItem.price, searchLevel)
    g_currentMission:addMoney(-fee, farmId, MoneyType.SHOP_VEHICLE_BUY, true, true)
    -- Log:var("Fee deducted", fee)
end

function BuyUsedEquipment:finalizeSearch(farmId, xmlFilename, success)
    if g_server == nil then
        Log:warning("finalizeSearch command is only allowed on the server")
        return
    end

    if success then
        local storeItem = g_storeManager:getItemByXMLFilename(xmlFilename)
        -- local farm = g_farmManager:getFarmById(farmId)

        self:generateSaleItem(storeItem)
        -- Log:debug("Sale item generated")
    end

    local evt = NotifySearchCompletedEvent.new(farmId, xmlFilename, success)

    g_server:broadcastEvent(evt, true)

    Log:debug("Clients notified")
end



function BuyUsedEquipment:generateSaleItem(storeItem, preferredGeneration)

    local function getRandomValue(pair)
        local minValue, maxValue = unpack(pair)
        return math.random() * (maxValue - minValue) + minValue
    end

    -- local preferredGeneration = 4 --TODO: remove
    local generationIndex = preferredGeneration or math.random(self.MIN_GENERATION, self.MAX_GENERATION)
    local generation = GENERATIONS[generationIndex]
    local wear = getRandomValue(generation.wear)
    local damage = getRandomValue(generation.damage)
    local hours = getRandomValue(generation.hours)
    local operatingTime = hours * (60 * 60 * 1000)
    local age = getRandomValue(generation.age)
    local discount = getRandomValue(generation.discount)
    local priceFactor = (1 - discount)
    local boughtConfigurations = {}

    local price = storeItem.price * priceFactor

    if self.USE_ALT_PRICE_STRATEGY then
        local defaultPrice = StoreItemUtil.getDefaultPrice(storeItem, boughtConfigurations)
        local repairPrice = Wearable.calculateRepairPrice(defaultPrice, damage)
        local repaintPrice = Wearable.calculateRepaintPrice(defaultPrice, wear)
        local altPrice = Vehicle.calculateSellPrice(storeItem, age, operatingTime, defaultPrice, repairPrice, repaintPrice)
        local deltaFactor = 1 + (math.random(-self.ALT_PRICE_DELTA, self.ALT_PRICE_DELTA) / 100)
        -- Log:var("alt price", altPrice)

        price = altPrice * deltaFactor
    end

    g_currentMission.vehicleSaleSystem:addSale({
		["timeLeft"] = math.random(self.MIN_SALE_DURATION, self.MAX_SALE_DURATION),
		["isGenerated"] = false,
		["xmlFilename"] = storeItem.xmlFilename,
		["boughtConfigurations"] = boughtConfigurations,
		["age"] = age,
		["price"] = price,
		["damage"] = damage,
		["wear"] = wear,
		["operatingTime"] = operatingTime,
	})

    -- Log:debug("Item added")
end
